/*
 *  Copyright 2014 Embedded Artists AB
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
 
#ifndef BUTTON_H
#define BUTTON_H

#include "Clickable.h"

/**
 * A button with a caption
 *
 * @code
 * #include "mbed.h"
 * #include "Button.h"
 *
 * SWIM_WINDOW_T win;
 *
 * static void buttonClicked(uint32_t x)
 * {
 *   bool* done = (bool*)x;
 *   *done = true;
 * }
 * 
 * int main(void) {
 *    // initialize the display and touch
 *    DMBoard::instance().init();
 *    
 *    // setup the SWIM window to use
 *    swim_window_open(&win, ...);
 *
 *    // create a 60x30 pixels button labeled "Done" at 100,100
 *    Button btn("Done", win.fb, 100, 100, 60, 30);
 *    btn.draw();
 *
 *    // register a callback for when the button is pressed and pass the
 *    // done flag for the callback to modify
 *    bool done = false;
 *    btn.setAction(buttonClicked, (uint32_t)&done);
 *
 *    // keep processing touch events until the button is clicked
 *    TouchPanel* touch = DMBoard::instance().touchPanel();
 *    TouchPanel::touchCoordinate_t coord;
 *    while(!done) {
 *      if (touch->read(coord) == TouchPanel::TouchError_Ok) {
 *        if (btn.handle(coord.x, coord.y, coord.z > 0)) {
 *          btn.draw();
 *        }
 *      }
 *    }
 * }
 * @endcode
 */
class Button : public Clickable {
public:

    /** Creates a new button
     *
     *  This button will use a SWIM window to draw on. That window will use
     *  part of the full size frame buffer to draw on.
     *
     *  @param caption the button text
     *  @param fb      the frame buffer
     *  @param x       the upper left corner of the button
     *  @param y       the upper left corner of the button
     *  @param width   the width of the button
     *  @param height  the height of the button
     */
  Button(const char* caption, COLOR_T* fb, uint16_t x, uint16_t y, uint16_t width, uint16_t height);
  virtual ~Button() {};
  
    /** Changes the caption
     *
     *  @param caption the new text on the button
     */
  void setCaption(const char* caption);

    /** Changes the colors
     *
     *  @param bg        background color when not pressed
     *  @param fg        text color when pressed 
     *  @param bgPressed background color when pressed
     *  @param fgPressed text color when pressed
     */
  void setColors(COLOR_T bg, COLOR_T fg, COLOR_T bgPressed, COLOR_T fgPressed);
  
    /** Draws the button (on a new framebuffer if one is specified)
     *  @param fb      the frame buffer
     */
  virtual void draw(COLOR_T* fb = 0);

private:
  const char* _caption;
  int _capx, _capy;
  COLOR_T _bgCol, _fgCol, _bgColPressed, _fgColPressed;
};

#endif /* BUTTON_H */
