/*
 *  Copyright 2014 Embedded Artists AB
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */


#include "mbed.h"
#include "EthernetInterface.h"
#include "AppColorPicker.h"
#include "lpc_swim_font.h"


/******************************************************************************
 * Defines and typedefs
 *****************************************************************************/
 
#define BOX_SIDE   192
 
#define BTN_OFF    20
 
/******************************************************************************
 * Global variables
 *****************************************************************************/

/******************************************************************************
 * Private functions
 *****************************************************************************/

static void buttonClicked(uint32_t x)
{
  bool* done = (bool*)x;
  *done = true;
}

void AppColorPicker::draw()
{
    // Prepare fullscreen
    swim_window_open(_win, 
                   _disp->width(), _disp->height(),         // full size
                   _fb,
                   0,0,_disp->width()-1, _disp->height()-1, // window position and size
                   1,                                       // border
                   WHITE, WHITE, BLACK);                    // colors: pen, backgr, forgr
    swim_set_title(_win, "Color Picker", BLACK);

    swim_window_open(_colorwin, 
                     _disp->width(), _disp->height(),         // full size
                     _fb,
                     50,(_disp->height()-BOX_SIDE)/2,50+BOX_SIDE-1, BOX_SIDE+(_disp->height()-BOX_SIDE)/2,                 // window position and size
                     0,                                       // border
                     WHITE, WHITE, BLACK);                    // colors: pen, backgr, forgr
    

    uint16_t r, g, b;
    uint16_t rx = BOX_SIDE/32;
    uint16_t bx = BOX_SIDE/32;
    uint16_t gy = BOX_SIDE/64;
    uint16_t color;
    for (int x = 0; x < BOX_SIDE; x++) {
        r = (x/rx);
        b = 0x1f - (x/bx);
        color = ((r & 0x1f) << 11) | ((b & 0x1f) << 0);
        for (int y = 0; y < BOX_SIDE; y++) {
            g = (y/gy);
            _colorwin->pen = color | ((g & 0x3f) << 5);
            swim_put_pixel(_colorwin, x, y);
        }
    }
    
    _btn = new ImageButton(_win->fb, _win->xpmax - BTN_OFF - _resOk->width(), _win->ypmax - BTN_OFF - _resOk->height(), _resOk->width(), _resOk->height());
    _btn->loadImages(_resOk);
    _btn->draw();
    
    // Copy everything onto the back buffer
    memcpy(_fb2, _fb, _disp->fbSize());
}

/******************************************************************************
 * Public functions
 *****************************************************************************/

AppColorPicker::AppColorPicker() : _disp(NULL), _win(NULL), _colorwin(NULL), 
  _fb(NULL), _fb2(NULL), _btn(NULL), _resOk(NULL)
{
    if (DMBoard::instance().display()->width() == 480) {
        _resultX = 350; 
        _resultY = 70; 
        _resultW = 80; 
        _resultH = 80;
    } else {
        _resultW = 80*2; 
        _resultH = 80*2;
        _resultX = 350;
        _resultY = (DMBoard::instance().display()->height() - (_resultH + 50))/2; 
    }
}

AppColorPicker::~AppColorPicker()
{
    teardown();
}

bool AppColorPicker::setup()
{
    _disp = DMBoard::instance().display();
    _win = (SWIM_WINDOW_T*)malloc(sizeof(SWIM_WINDOW_T));
    _colorwin = (SWIM_WINDOW_T*)malloc(sizeof(SWIM_WINDOW_T));
    _fb = (COLOR_T*)_disp->allocateFramebuffer();
    _fb2 = (COLOR_T*)_disp->allocateFramebuffer();

    return (_win != NULL && _colorwin != NULL && _fb != NULL && _fb2 != NULL);
}

void AppColorPicker::runToCompletion()
{
    bool done = false;
    draw();
    _btn->setAction(buttonClicked, (uint32_t)&done);
    void* oldFB = _disp->swapFramebuffer(_fb2);
    
    // Wait for touches
    TouchPanel* touch = DMBoard::instance().touchPanel();
    touch_coordinate_t coord;
    char buf[10];
    swim_set_pen_color(_win, BLACK);
    bool showingFB2 = true;
    COLOR_T lastColor = WHITE;
    while(!done) {
      // wait for a new touch signal (signal is sent from AppLauncher,
      // which listens for touch events)
      Thread::signal_wait(0x1);
      if (touch->read(coord) == TouchPanel::TouchError_Ok) {
        if (coord.z > 0 &&
            coord.x >= _colorwin->xpmin && coord.x <= _colorwin->xpmax && 
            coord.y >= _colorwin->ypmin && coord.y <= _colorwin->ypmax) {
            int x = coord.x - _colorwin->xpmin;
            int y = coord.y - _colorwin->ypmin;
            COLOR_T c = ((x/(BOX_SIDE/32))<<11) | ((y/(BOX_SIDE/64))<<5) | ((0x1f-(x/(BOX_SIDE/32)))<<0);
          if (c != lastColor) {
            swim_set_fill_color(_win, c);
            swim_put_box(_win, _resultX, _resultY, _resultX+_resultW, _resultY+_resultH);
            sprintf(buf, "0x%04x  ", c);
            swim_put_text_xy(_win, buf, _resultX, _resultY+_resultH+10);
                
            // Swap what is shown and what is drawn on
            if (showingFB2) {
              _disp->setFramebuffer(_fb);
              _win->fb = _fb2;
            } else {
              _disp->setFramebuffer(_fb2);
              _win->fb = _fb;
            }
            Thread::wait(20);
            showingFB2 = !showingFB2;
            lastColor = c;
          }
        }
        if (_btn->handle(coord.x, coord.y, coord.z > 0)) {
            _btn->draw(showingFB2 ? _fb2 : _fb);
        }
      }
    }
    
    // User has clicked the button, restore the original FB
    _disp->swapFramebuffer(oldFB);
    swim_window_close(_win);
    swim_window_close(_colorwin);
}

bool AppColorPicker::teardown()
{
    if (_win != NULL) {
        free(_win);
        _win = NULL;
    }
    if (_colorwin != NULL) {
        free(_colorwin);
        _colorwin = NULL;
    }
    if (_fb != NULL) {
        free(_fb);
        _fb = NULL;
    }
    if (_fb2 != NULL) {
        free(_fb2);
        _fb2 = NULL;
    }
    if (_btn != NULL) {
        delete _btn;
        _btn = NULL;
    }
    return true;
}

void AppColorPicker::addResource(Resources id, Resource* res)
{
    _resOk = res;
}
