/*
 *  Copyright 2014 Embedded Artists AB
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
 
#ifndef RENDERER_H
#define RENDERER_H

#include "rtos.h"
#include "Display.h"

/**
 * Renderer for the SlideShow engine.
 *
 * The renderer handles all (if more than one) running slideshows and
 * the layer(s) they are drawn on.
 *
 * For information on how to use the SlideShow and some examples see
 * https://developer.mbed.org/teams/Embedded-Artists/wiki/LPC4088DM-Using-the-SlideShow-Engine
 */
class Renderer {
public:

    Renderer();
    ~Renderer();

    /** Registers a part of the screen on a specific layer to be in use.
     *
     *  Returns a handle to pass when updating the framebuffer.
     *
     *  @param layer  0 is the bottom of the stack, higher number is on top
     *  @param xoff   top left corner of the drawing rectangle
     *  @param yoff   top left corner of the drawing rectangle
     *  @param width  width of the drawing rectangle
     *  @param height height of the drawing rectangle
     *
     *  @returns
     *       handle to pass to setFrameBuffer function
     *       0 on failure
     */
    uint32_t registerUser(int layer, int xoff, int yoff, int width, int height);

    /** Registers the entire screen on a specific layer to be in use.
     *
     *  Returns a handle to pass when updating the framebuffer.
     *
     *  @param layer  0 is the bottom of the stack, higher number is on top
     *
     *  @returns
     *       handle to pass to setFrameBuffer function
     *       0 on failure
     */
    uint32_t registerFullscreenUser(int layer);

    /** Removes a previously registered user
     *
     *  @param handle the handle from the registerUser() call
     */
    void unregisterUser(uint32_t handle);

    /** Informs the renderer that there is new data to render
     *
     * Blocks until the data has been registered. After that point the
     * data must not be modified until next call to setFramebuffer.
     *
     *  @param handle the handle from the registerUser() call
     *  @param data   the image data
     */
    void setFramebuffer(uint32_t handle, const uint16_t* data);

    /** Run the renderer
     *
     * Should be called from a high priority thread.
     */
    void render();

private:

    enum Constants {
      MaxNumLayers = 10,
    };

    typedef uint16_t* image_t;

    typedef struct {
      bool used;
      int x0;
      int y0;
      int x1;
      int y1;
      int width;
      int clippedHeight;
      int zorder;
      int signalId;
      bool fullscreen;
      const uint16_t* newData;
      const uint16_t* activeData;
      uint16_t* tmp;
      Mutex* lock;
    } layerinfo_t;

    layerinfo_t layers[MaxNumLayers];
    layerinfo_t* order[MaxNumLayers];

    int numRegisteredLayers;

    osThreadId threadId;
    Mutex setupMutex;

    int screenWidth;
    int screenHeight;
    int screenPixels;
    int screenBytes;
    int activeBackBuffer;

    image_t ImageBackBuffer[2];

    Display* display;
};

#endif



