#include "Rede/EthernetIf.h"

#include "lwip/inet.h"
#include "lwip/netif.h"
#include "netif/etharp.h"
#include "lwip/dhcp.h"
#include "arch/lpc17_emac.h"
#include "lpc_phy.h"
#include "lwip/tcpip.h"

#include "mbed.h"

struct netif EthernetIf::lpcNetif;
bool EthernetIf::use_dhcp;
ip_addr_t EthernetIf::ip_n;
ip_addr_t EthernetIf::netmask_n;
ip_addr_t EthernetIf::gw_n;
Semaphore EthernetIf::tcpip_initialized(0);
Semaphore EthernetIf::link_up(0);
Semaphore EthernetIf::netif_up(0);
bool EthernetIf::link_is_up(false);

uint32_t EthernetIf::os_timer_cb_led[5]; 
osTimerDef_t EthernetIf::os_timer_def_led = { (led_refresh), (os_timer_cb_led) };

void EthernetIf::tcpip_init_done(void *arg)
{
    //This function is called when the TCPIP stack finishes to initialize
    tcpip_initialized.release();
}

void EthernetIf::link_status_callback(struct netif *netif)
{
    //Called when the interface link goes up or down
    if(netif_is_link_up(netif))
    {
        printf("Link up!!!\n");
        if(!link_is_up)
        {
            if(use_dhcp)
            {
                dhcp_start(&lpcNetif);
            }
            else
            {
                netif_set_addr(&lpcNetif,&ip_n,&netmask_n,&gw_n);
                netif_set_up(&lpcNetif);
            }
        }
        link_up.release();
        link_is_up = true;
    }
    else
    {
        printf("Link down!!!\n");
        netif_set_down(&lpcNetif);
        link_is_up = false;
    }
}

void EthernetIf::interface_status_callback(struct netif *netif)
{
    //called when the status of the interface is up or down
    if(netif_is_up(netif))
    {
        netif_up.release();
        printf("Interface Ethernet is up\n");
        printf("IP Address: %s\n",get_IpAddress());
        printf("Netmask: %s\n",get_Netmask());
        printf("Gateway: %s\n\n",get_Gateway());
    }
}

void EthernetIf::led_refresh(void const* arg)
{
    if(LPC_GPIO1->FIOPIN & (1<<25))
        LPC_GPIO0->FIOSET |= (1<<4);
    else
        LPC_GPIO0->FIOCLR |= (1<<4);

    if(LPC_GPIO1->FIOPIN & (1<<26))
        LPC_GPIO0->FIOSET |= (1<<5);
    else
        LPC_GPIO0->FIOCLR |= (1<<5);
}

int EthernetIf::_init()
{

    LPC_PINCON->PINSEL3 &= ~(3<<18);    //GPIO P1.25 - MBED LED LINK
    LPC_PINCON->PINSEL3 &= ~(3<<20);    //GPIO P1.26 - MBED LED SPEED
    LPC_PINCON->PINSEL0 &= ~(3<<8);     //GPIO P0.4 - ETHERNET GREEN LED
    LPC_PINCON->PINSEL0 &= ~(3<<10);    //GPIO P0.5 - ETHERNET YELLOW LED
    LPC_GPIO1->FIODIR &= ~((1<<25)|(1<<26));
    LPC_GPIO0->FIODIR |= ((1<<4)|(1<<5));
    
    tcpip_init(tcpip_init_done, NULL);
    tcpip_initialized.wait();
    
    memset((void*) &lpcNetif, 0, sizeof(lpcNetif));
    netif_add(&lpcNetif, &ip_n, &netmask_n, &gw_n, NULL, lpc_enetif_init, tcpip_input);
    netif_set_default(&lpcNetif);
    
    netif_set_link_callback  (&lpcNetif, link_status_callback);
    netif_set_status_callback(&lpcNetif, interface_status_callback);
    
    return 0;
}

int EthernetIf::Initialize()
{
    use_dhcp = true;
    
    return _init();
}

int EthernetIf::Initialize(char *ip,char *netmask,char *gw)
{
    use_dhcp = false;
    inet_aton(ip,&ip_n);
    inet_aton(netmask,&netmask_n);
    inet_aton(gw,&gw_n);
    
    return _init();
}

int EthernetIf::Connect(unsigned int timeout /*= 1500*/)
{
    NVIC_SetPriority(ENET_IRQn, ((0x01 << 3) | 0x01));
    NVIC_EnableIRQ(ENET_IRQn);
    
    netif_set_link_up(&lpcNetif);
    int inited;
    inited = netif_up.wait(timeout);
    
    osTimerId led_timer = osTimerCreate(osTimer(led), osTimerPeriodic, NULL);
    osTimerStart(led_timer, 100);
    
    return (inited > 0) ? (0) : (-1);
}

int EthernetIf::Disconnect()
{
    if (use_dhcp)
    {
        dhcp_release(&lpcNetif);
        dhcp_stop(&lpcNetif);
    }
    else
    {
        netif_set_down(&lpcNetif);
    }
    
    NVIC_DisableIRQ(ENET_IRQn);
    
    return 0;
}

bool EthernetIf::is_LinkUp()
{
    if(netif_is_link_up(&lpcNetif))
        return true;
    else
        return false;
}

bool EthernetIf::is_InterfaceUp()
{
    if(netif_is_up(&lpcNetif))
        return true;
    else
        return false;
}

char* EthernetIf::get_IpAddress()
{
    return ipaddr_ntoa(&lpcNetif.ip_addr);
}

char* EthernetIf::get_Netmask()
{
    return ipaddr_ntoa(&lpcNetif.netmask);
}

char* EthernetIf::get_Gateway()
{
    return ipaddr_ntoa(&lpcNetif.gw);
}