/* mbed Microcontroller Library
 * Copyright (c) 2019 ARM Limited
 * SPDX-License-Identifier: Apache-2.0
 */

#include "mbed.h"
#include "platform/mbed_thread.h"
#include "stdio.h"

// Blinking rate in milliseconds
#define BLINKING_RATE_MS 500
#define SW2 P0_4
// Specify different pins to test printing on UART other than the console UART.
#define TARGET_TX_PIN USBTX
#define TARGET_RX_PIN USBRX

// Create a BufferedSerial object to be used by the system I/O retarget code.
static Serial serial_port(TARGET_TX_PIN, TARGET_RX_PIN, 115200);

// Global variables
char buffer[80];
Thread thread;
bool allowUpdate = true;

FileHandle *mbed::mbed_override_console(int fd)
{
    return &serial_port;
}

// Initialise the digital pin LED1 as an output
DigitalOut led(LED1); // Status indicator LED flashes at 1Hz
DigitalIn pushButton(SW2, PullUp);
AnalogIn vTherm(P10_1); // Thermistor temperature sensor
AnalogIn lightLevel(P10_4); //LDR light level sensor
DigitalOut boilerLed (P10_5); //indicator of boiler status
DigitalOut lightingLed (P10_0); // indicator of light switch status on/off
DigitalOut yellowLed (P0_5); // Todo

/* Data Structures */

struct dataSet {
    float highTempThresh = 26.0;
    float lowTempThresh = 23.0;
    float ambientTemp;
    float highLightThresh = 80.0;
    float lowLightThresh = 20.0;
    float ambientLight;
} myData;

/* prototype of function */
void displayAt( int x, int y, char *buffer );
void initialise();
void readSensors();
void setActuators();
void displayData();

/* console thread */
void consoleThread()
{
    char inputStr[80];
    int index = 0; // character input index in inputStr

    int selection = 0; //{tempLow==0, tempHigh==1, lightLow==2, lightHigh==3}
    while(1) {
        if (serial_port.readable()) {
            allowUpdate = false;
            switch ( selection ) {
                case 0:   //tempLow:
                    printf("\033[4;31H\033[1;30m\033[47m%2.1f\033[1;37m\033[40m\033[14;1H", myData.lowTempThresh);
                    break;
                case 1:   //tempHigh:
                    printf("\033[4;47H\033[1;30m\033[47m%2.1f\033[1;37m\033[40m\033[14;1H", myData.highTempThresh);
                    break;
                case 2:   //lightLow:
                    printf("\033[5;31H\033[1;30m\033[47m%2.1f\033[1;37m\033[40m\033[14;1H", myData.lowLightThresh);
                    break;
                case 3:  //lightHigh:
                    printf("\033[5;47H\033[1;30m\033[47m%2.1f\033[1;37m\033[40m\033[14;1H", myData.highLightThresh);
                    break;
            }
            char inputChar;
            inputChar = serial_port.getc();
            switch (inputChar) {
                case 0x08:  // backspace
                    
                    if (index > 0) inputStr[--index] = NULL;;
                    
                    printf("\033[14;1HChange to: %s %c", inputStr, 0x08);
                    printf("\033[25h");
                    break;
                case 0x09:  // tab key
                    switch (selection) { // currently selected threshold{
                        case 0:
                            printf("\033[4;31H\033[0;37m\033[40m%2.1f\033[4;47H\033[1;30m\033[47m%2.1f\033[1;37m\033[40m\033[14;1H", myData.lowTempThresh, myData.highTempThresh);
                            break;
                        case 1:
                            printf("\033[4;47H\033[0;37m\033[40m%2.1f\033[5;31H\033[1;30m\033[47m%2.1f\033[1;37m\033[40m\033[14;1H", myData.highTempThresh, myData.lowLightThresh);
                            break;
                        case 2:
                            printf("\033[5;31H\033[0;37m\033[40m%2.1f\033[5;47H\033[1;30m\033[47m%2.1f\033[1;37m\033[40m\033[14;1H", myData.lowLightThresh, myData.lowLightThresh);
                            break;
                        case 3:
                            printf("\033[5;47H\033[0;37m\033[40m%2.1f\033[4;31H\033[1;30m\033[47m%2.1f\033[1;37m\033[40m\033[14;1H", myData.lowLightThresh, myData.lowTempThresh);
                            break;
                    }
                    selection = (selection + 1)%4; // keep it within bounds
                    
                    break;
                case 0x0d:  // enter key, accept input
                    if (index > 0) {
                        switch (selection) {
                        case 0:
                            myData.lowTempThresh = atof(inputStr);
                            break;
                        case 1:
                            myData.highTempThresh = atof(inputStr);
                            break;
                        case 2:
                            myData.lowLightThresh = atof(inputStr);
                            break;
                        case 3:
                            myData.highLightThresh = atof(inputStr);
                            break;
                        }
                    }
                    inputStr[index] = NULL;
                    index = 0;

                    allowUpdate = true;
                    printf("\033[14;1H\033[2K");
                    printf("\033[?25l");
                    break;
                case 0x20: // Space bar data coming...
                    printf("\033[14;1HChange to: ");
                    break;
                default: // should be a digit or decimal point
                    inputStr[index++] = inputChar;
                    inputStr[index] = NULL;
                    printf("\033[14;1HChange to: %s", inputStr);
                    printf("\033[?25h");
                    break;
            }
        }
        thread_sleep_for(5); // let other task do stuff for 5ms
    }
}


int main()
{
    initialise(); // function to setup VT100 display
    thread.start( consoleThread );
    /****************************************************************************
     *
     * Main loop:-
     *
     * flash status led
     * read sensors
     * display data read from the sensors and threshold values
     * sleep for 0.5seconds
     *
     ***************************************************************************/

    while (true) {
        led = !led; // flashing status signal
        readSensors();
        setActuators();
        displayData();
        thread_sleep_for(BLINKING_RATE_MS);
    }
}
void setActuators() {
    if (myData.ambientLight > myData.lowLightThresh) lightingLed = true; // Light on when light level beneath thresh
    if (myData.ambientLight > myData.highLightThresh) lightingLed = false; // Light off when light level above thresh
    if (myData.ambientTemp > myData.lowTempThresh) lightingLed = true; // boiler on when temp below thresh
    if (myData.ambientTemp > myData.highTempThresh) lightingLed = false; // boiler off when above thresh
}
void readSensors()
{
    /* read thermistor Voltage */
    float refVoltage = vTherm.read() * 2.4; // Range of ADC 0->2*Vref
    float refCurrent = refVoltage  / 10000.0; // 10k Reference Resistor
    float thermVoltage = 3.3 - refVoltage;    // Assume supply voltage is 3.3v
    float thermResistance = thermVoltage / refCurrent;
    float logrT = (float32_t)log((float64_t)thermResistance);

    /* Calculate temperature from the resistance of thermistor using Steinhart-Hart Equation */
    float stEqn = (float32_t)((0.0009032679) + ((0.000248772) * logrT) +
                              ((2.041094E-07) * pow((float64)logrT, (float32)3)));

    myData.ambientTemp = (float32_t)(((1.0 / stEqn) - 273.15)  + 0.5);
    myData.ambientLight = ( 1 - lightLevel.read()) * 100;
}
void displayData()
{
    int tCol, lCol;
    if (myData.ambientTemp > myData.highTempThresh) {
        tCol=41;    //Red Text
        printf("\033[1;31m");
    } else if (myData.ambientTemp < myData.lowTempThresh) {
        tCol=44;    // Blue text
        printf("\033[1;34m");
    } else  {
        tCol=42;    // Green Text
        printf("\033[1;32m");
    }
    sprintf(buffer, "Temperature is:   %2.1fC ", myData.ambientTemp);
    displayAt(1, 4, buffer);
    printf("\033[0;37m");
    sprintf(buffer, "%2.1fC ", myData.lowTempThresh);
    displayAt(31, 4, buffer);
    sprintf(buffer, "%2.1fC ", myData.highTempThresh);
    displayAt(47, 4, buffer);
    sprintf(buffer,"\033[%dm \033[40m", tCol);
    displayAt(26, 4, buffer);
    sprintf( buffer, "The boiler is %s", boilerLed?"On ":"Off" );
    displayAt(1, 6, buffer);

    if (myData.ambientLight > myData.highLightThresh) {
        lCol=41;    //Red Text
        printf("\033[1;31m");
    } else if (myData.ambientLight < myData.lowLightThresh) {
        lCol=44;    // Blue text
        printf("\033[1;34m");
    } else {
        lCol=42;    // Green Text
        printf("\033[1;32m");
    }
    sprintf( buffer, "Ambient Light is: %3.1f%c ", myData.ambientLight, '%' );
    displayAt(1, 5, buffer);
    sprintf(buffer,"\033[%dm \033[40m", lCol);
    displayAt(26, 5, buffer);
    printf("\033[0;37m");
    sprintf(buffer, "%2.1f%c ", myData.lowLightThresh, 0x25);
    displayAt(31, 5, buffer);
    sprintf(buffer, "%2.1f%c ", myData.highLightThresh, 0x25);
    displayAt(47, 5, buffer);
    sprintf( buffer, "The lights are %s", lightingLed?"On ":"Off" );
    displayAt(1, 7, buffer);
}

void displayAt( int x, int y, char *buffer )
{
    if (allowUpdate) {
        printf( "\033[%d;%dH%s", y, x, buffer);
        fflush(stdout);
    }
}

void initialise()
{
    printf("\033[2J\033[H"); // clear screen and move the cursor to 0, 0
    printf("\033[?25l"); // Turn off visible cursor[?25 lower case L


    /* what is the character set?

        for (int i = 32; i < 255; i ++) {
            printf(" %c ", i);
        }
        thread_sleep_for(10000);
    */
    printf("Environmental Control System");
    printf( "\033[1;37m" );
    printf("\033[3;28H\033[1;34mLow Threshold  \033[1;31mHigh Threshold");
    printf("\033[2;37m\033[16;1H* Press \"Space key\" to set threshold values\r\n");
    printf("  Use \"Tab key\" to select each threshold setting\r\n");
    printf("  Hit \"Enter key\" to store new threshold value");
    fflush(stdout); // send the codes to the terminal
}