/* mbed Microcontroller Library
 * Copyright (c) 2019 ARM Limited
 * SPDX-License-Identifier: Apache-2.0
 */

#include "mbed.h"
#include "platform/mbed_thread.h"
#include "stdio.h"

// Blinking rate in milliseconds
#define BLINKING_RATE_MS 500
#define SW2 P0_4
// #define alsOut P10_0

//Global variables
static Serial serial_port(USBTX, USBRX, 115200);
char buffer[80];
Thread thread;

FileHandle *mbed::mbed_override_console(int fd)
{

    return &serial_port;
}
// Initialise the digital pin LED1 as an output
DigitalOut led(LED1);
DigitalIn pushButton(SW2, PullUp);
DigitalOut thermVcc(P10_3);
DigitalOut thermGnd(P10_0);
AnalogIn thermVal(P10_1);
AnalogIn lightLevel(P10_4);

/* Data Structures */

struct dataSet {
    float highTempThresh = 26;
    float lowTempThresh = 23;
    float ambientTemp;
    float highLightThresh = 80;
    float lowLightThresh = 20;
    float ambientLight;
} myData;

/* prototype of function */
void displayAt( int y, int x, char *buffer );
void initialise();
void readSensors();
void displayData();

void consoleThread()
{
    char inputChar;
    int index =0;
    while ( true ) {
        if (serial_port.readable()) {
            inputChar = serial_port.getc();
            index++;
            printf("\033[14;1H%c %d", inputChar, index);
        }
        thread_sleep_for(5);
    }
}

int main()
{
    initialise(); // function to setup VT100 display
    thread.start ( consoleThread ); // start the console keyboard scanning thread running

    /******************************************************************************************
    *
    * Main loop:=
    *
    * flash status led
    * read sensors
    * display data read from the sensors and threshold values
    * sleep for 0.5seconds
    *
    *******************************************************************************************/

    while (true) {
        led=!led; // flashing status signal
        readSensors();
        displayData();
        thread_sleep_for(BLINKING_RATE_MS);
    }
}
void readSensors()
{
    /*read thermistor Voltage*/
    float refVoltage = thermVal.read() * 2.4;// Range of ADC 0->2*Vref
    float refCurrent = refVoltage / 10000.0; //10k Reference Resistor
    float therVoltage = 3.3 - refVoltage; // Assume supply voltage is 3.3v
    float thermResistance = therVoltage / refCurrent;
    float logrT = (float32_t)log((float64_t)thermResistance);

    /*Calculate temperature from the resistance of thermistor using Steinhart-Hart Equation*/
    float stEqn = (float32_t) ((0.0009032679) + ((0.000248772) * logrT) + ((2.041094E-07) * pow((float64)logrT, (float32)3)));

    myData.ambientTemp = (float32_t)(((1.0 / stEqn) - 273.15) + 0.5);
    myData.ambientLight = lightLevel.read() * 100;
}
void displayData()
{

    int tCol, lCol;
    if (myData.ambientTemp >  myData.highTempThresh) {
        printf("\033[31m");  //Red Text
        tCol = 41;
    } else if (myData.ambientTemp >  myData.lowTempThresh) {
        printf("\033[34m");  //Blue Text
        tCol = 44;
    } else {
        printf("\033[32m"); // Green Text
        tCol = 42; // set background to green
    }
    sprintf(buffer, "Temerpature is %2.1f C", myData.ambientTemp);
    displayAt (1, 4, buffer);
    sprintf(buffer, "%2.1f C", myData.lowTempThresh);
    printf("\033[37m"); // default text
    displayAt (31, 4, buffer);
    sprintf(buffer, "%2.1f C", myData.highTempThresh);
    displayAt (47, 4, buffer);
    printf("\033[4;25H\033[%dm \033[40m", tCol); // print a simulated led

    if (myData.ambientLight >  myData.highLightThresh) {
        printf("\033[31m");  //Red Text
        lCol = 41;
    } else if (myData.ambientLight >  myData.lowLightThresh) {
        printf("\033[34m");  //Blue Text
        lCol = 44;
    } else {
        printf("\033[32m"); // Green Text
        lCol = 42; // set background to green
        }
        sprintf( buffer, "Ambient Light is: %3.1f%c", myData.ambientLight, '%' );
        displayAt(1,5, buffer);
        printf("\033[37m"); // default text
        sprintf(buffer, "%2.1f%c", myData.lowLightThresh, '%' );
        displayAt (31, 5, buffer);
        sprintf(buffer, "%2.1f%c", myData.highLightThresh, '%' );
        displayAt (47, 5, buffer);
        printf("\033[4;25H\033[%dm \033[40m", lCol);
    }
    void displayAt( int y, int x, char *buffer ) {
        printf("\033[%d;%dH%s", y, x, buffer);
        fflush(stdout);
    }

    void initialise() {
        printf("\033[2J\033[H"); // clear screen and move the cursor to 0, 0
        printf("\033[?25l"); // Turn off visable cursor
        printf("Environmental Control System");
// printf( "|033[37m" );
        printf("\033[3;28H\033[1;34mLow Threshold \033[1;31mHigh Threshold");
        printf("\033[1;37m\033[16;1H* Press \"space bar to change threshold levels\r\n");
               fflush(stdout); // send the codes to the terminal
    }