/*******************************************************************************
* Copyright (C) 2017 Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
*******************************************************************************/

#ifndef MaximInterface_DS28EC20
#define MaximInterface_DS28EC20

#include "MaximInterface/Links/SelectRom.hpp"
#include "MaximInterface/Links/Sleep.hpp"
#include "MaximInterface/Utilities/Export.h"

namespace MaximInterface {

///
/// DS28EC20 20480-bit 1-Wire EEPROM
///
/// @details
/// The  DS28EC20  is  a  20480-bit,  1-Wire (R)   EEPROM
/// organized as 80 memory pages of 256 bits each. An
/// additional  page  is  set  aside  for  control  functions.
/// Data is written to a 32-byte scratchpad, verified, and
/// then  copied  to  the  EEPROM  memory.  As  a  special
/// feature,  blocks of eight memory pages can  be  write
/// protected  or  put  in  EPROM-Emulation  mode,  where
/// bits can only be changed from a 1 to a 0 state. The
/// DS28EC20 communicates over the single-conductor
/// 1-Wire bus. The communication follows the standard
/// 1-Wire protocol. Each device has its own unalterable
/// and  unique  64-bit  ROM  registration  number.  The
/// registration number is used to address the device in
/// a multidrop 1-Wire net environment.
///


class DS28EC20
{
    public:
        typedef array<uint_least8_t, 32u> Scratchpad;
        typedef uint_least16_t Address;

        enum ErrorValue
        {
            CrcError = 1,
            OperationFailure
        };

    public:
        static const uint_least8_t familyCode = 0x43u;

        static const size_t pageSizeBytes = 32u;

        /// Number of memory pages on the device
        /// 2560 bytes / 32 bytes/page = 80 pages
        static const size_t memoryPages = 80u;

        DS28EC20(const Sleep & sleep, OneWireMaster & master, const SelectRom & selectRom, RomId::const_span romId)
                : selectRom(selectRom), romId(&romId), master(&master), sleep(&sleep)
        {
        }

        void setSleep(const Sleep & sleep)
        {
            this->sleep = &sleep;
        }
        void setMaster(OneWireMaster & master)
        {
            this->master = &master;
        }
        void setSelectRom(const SelectRom & selectRom)
        {
            this->selectRom = selectRom;
        }
        void setRomId(RomId::const_span & romId)
        {
            this->romId = &romId;
        }

        /// Reads block of data from EEPROM memory.
        /// @param[in] beginAddress EEPROM memory address to start reading from.
        /// @param[out] data EEPROM data read from the device.
        /// @param[in] dataLen Length of data parameter and number of bytes to read.
        MaximInterface_EXPORT
        error_code readMemory(Address beginAddress, uint_least8_t * data, size_t dataLen) const;

        /// Reads block of data from EEPROM memory and checks the CRC at the end.
        /// @param[in] beginAddress EEPROM memory address to start reading from.
        /// @param[out] data EEPROM data read from the device.
        /// @param[in] dataLen Length of data parameter and number of bytes to read.
        MaximInterface_EXPORT
        error_code readMemoryExt(Address beginAddress, uint_least8_t * data, size_t dataLen) const;

        /// Writes 8 bytes to the scratchpad.
        /// @param[in] targetAddress EEPROM memory address that this data.
        /// will be copied to.  Must be on row boundary.
        /// @param[in] data Data to write to scratchpad.
        MaximInterface_EXPORT
        error_code writeScratchpad(Address targetAddress, const Scratchpad & data);

        /// Reads contents of scratchpad.
        /// @param[out] data Data read from scratchpad.
        /// @param[out] esByte E/S byte read before scratchpad data.
        MaximInterface_EXPORT
        error_code readScratchpad(Scratchpad & data, uint_least8_t & esByte);

        /// Copies contents of scratchpad to EEPROM.
        /// @param[in] targetAddress EEPROM memory address that scratchpad
        /// will be copied to. Must be on row boundary.
        /// @param[in] esByte E/S byte from preceding Read Scratchpad command.
        MaximInterface_EXPORT
        error_code copyScratchpad(Address targetAddress, uint_least8_t esByte);

        MaximInterface_EXPORT
        static const error_category & errorCategory();

    private:
        SelectRom selectRom;
        RomId::const_span * romId;
        OneWireMaster * master;
        const Sleep * sleep;
};

/// Writes data to EEPROM using Write Scratchpad, Read Scratchpad,
/// and Copy Scratchpad commands.
/// @param[in] targetAddress EEPROM memory address to start writing at.
/// @param[in] data Data to write to EEPROM.
MaximInterface_EXPORT error_code writeMemory(DS28EC20 & device, DS28EC20::Address targetAddress, const DS28EC20::Scratchpad & data);

/// Writes data to EEPROM using Write Scratchpad, Read Scratchpad,
/// and Copy Scratchpad commands.
/// @param[in] targetAddress EEPROM memory address to start writing at.
/// @param[in] data Data to write to EEPROM.
/// @param[in] data Length of data to be written.
MaximInterface_EXPORT error_code writeMemory(DS28EC20 & device, DS28EC20::Address targetAddress, const uint_least8_t * dataIn, size_t dataLen);

inline error_code make_error_code(DS28EC20::ErrorValue e)
{
    return error_code(e, DS28EC20::errorCategory());
}

} // namespace MaximInterface

#endif
