/* mbed EventFramework Library
 * Copyright (c) 2012-2016 raulMrello
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef _EVENTHANDLER_H_
#define _EVENTHANDLER_H_

#include "../Types/Types.h"

/** EventHandler class
 *
 * EventHandler instances are the EventFramework entities which will process published
 * events. Each handler has a fixed priority in the range: 0(max) to 65535(min) that the
 * scheduler will use to fire its EventDispatchingRoutine when the event they are listening
 * has been published.  * Each EventHandler can listen to several events, through one or 
 * various EventDispatchingRoutines.
 *
 */
class EventHandler{
    public:    
    /** Creates an EventHandler with up to two arguments: the event dispatching function 
     * and a fixed priority
     *
     * @param func Dispatching routine to be invoked to process a published event.
     * @param data event handler attached data object reference
     * @param prio EventHandler priority in the range: 0[max] - 65535[min].
     */
    EventHandler(uint16_t prio, EventDispatchingRoutine* func=NULL, void* data=NULL);

    ~EventHandler();
    
    /** Gets the EventHandler priority.
     *
     * @returns EventHandler priority
     */
    uint16_t GetPrio();

    /** Executes its dispatching function to process a published event. 
     *
     * @param args data reference to be notified about the event processing.
     * @returns Error code in uint32_t format or 0 if success.
     */
    uint32_t Execute(void * args);

    /** Attaches a dispatching function to this handler, if NULL is passed, then has no effect.
     *
     * @param func Dispatching function to be invoked for event processing.
     */
    void Attach(EventDispatchingRoutine* func);
        
    protected:
    uint32_t(*func)(void*, void*);
    uint16_t prio;
    void* data;
};

#endif
