/* mbed EventFramework Library
 * Copyright (c) 2012-2016 raulMrello
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef _EVENT_H_
#define _EVENT_H_

#include "../List/List.h"
#include "../EventHandler/EventHandler.h"

/** Event class
 *
 * Events are the inter-process communication mechanism within the EventFramework. Each 
 * event registered in the framework stands for a type of signal, which can accept 
 * attached data to be processed by any processing entity; in this case by EventHandlers
 * through its attached EventDispatchingRoutine.
 * Each event has a fixed priority in the range: 0 (max) to 65535 (min) and can keep track
 * of a list of installed EventHandlers to invoke when the event is published. Published
 * events are queued into a PendingList managed by the EventFramework scheduler, who will
 * dispatch them accordingly with their priority (from highest to lowest).
 */
class Event{
    public:
    /** Creates an Event with a specified priority
     *
     * @param prio Event priority. Range 0[max] - 65535[min]
     */
    Event(uint16_t prio);

    ~Event();
    
    /** Gets the Event priority.
     *
     * @returns event priority
     */
    uint16_t GetPrio(void);

    /** Gets the base event reference
     *
     * To explain the way in which a base reference is used, it is necessary to explain what
     * happens on event publishing. Within the framework, only registered events can be published. 
     * while publishing an event of one type, a new event is allocated (by: new Event()) and it is referenced
     * to the existing one by its private [base] property. This base reference allows the scheduler to invoke
     * to all its listeners.
     *
     * @returns event reference to the base event.
     */
    Event*   GetBase(void);

    /** Sets the Event reference to which this is based from.
     *
     * @param ev event reference to link with.
     */
    void     SetBase(Event* ev);

    /** Sets attached data to the Event.
     *
     * Events can attach extra data to be processed by the listener. Through this interface, a data
     * reference can be attached to the event.
     *
     * @param data data reference to be attached to the event for further processing.
     */
    void     SetData(void* data);

    /** Gets event's attached data.
     *
     * @returns reference to the attached data.
     */
    void*    GetData(void);

    /** Adds an EventHandler to process this kind of events.
     *
     * Each event manages an internal list of handlers. If an event of this kind is published
     * all handlers in this list will be fired to process the event.
     *
     * @param hnd EventHandler reference to subscribe to this Event.
     */
    void     Subscribe(EventHandler* hnd);

    /** Erases an EventHandler to stop processing this kind of events.
     *
     * @param hnd EventHandler reference to unsubscribe from this Event.
     */
    void     Unsubscribe(EventHandler* hnd);

    /** Gets the list of subscribed EventHandlers
     *
     * @returns reference to the list of subscribed EventHandlers.
     */
    List*    GetList(void);
    
    private:
    uint16_t prio;
    void* data;
    List* hlist;
    Event* base;
};

#endif
