/**
 * @file    Adis16488.h
 * @brief   Device driver - ADIS16488 IMU
 * @author  sam grove
 * @version 1.0
 * @see     http://www.analog.com/static/imported-files/data_sheets/ADIS16488.pdf
 *
 * Copyright (c) 2013
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
 
#ifndef ADIS16488_H
#define ADIS16488_H

#include "LogUtil.h"
#include "mbed.h"

/** Using the Analog Devices ADIS16488/PCBZ
 *
 * Example:
 * @code
 *  #include "mbed.h"
 *
 *  int main()
 *  {
 *  }
 * @endcode
 */

/**
 *  @class Adis16488
 *  @brief API abstraction for the ADIS16488 IMU
 */ 
class Adis16488
{
public:

    struct{ uint16_t data[6]; } gyro, accel, magn, deltang, deltvel;
    
    /** Create the Adis16488 object
     *  @param spi - A defined SPI object
     *  @param cs - A defined DigitalOut object
     *  @param rst - A defined DigitalOut object
     *  @param dr - A defined InterruptIn object
     */  
    Adis16488(SPI &spi, DigitalOut &cs, DigitalOut &rst, InterruptIn &dr);
    
    /** Clear state vars and initilize the dependant objects
     */
    void init(void);
    
    /** Read from a register (exposed for debugging reasons)
     *  @param reg - The register to be written
     *  @param data - Data read from the device is stored here
     */
    void readRegister(uint16_t const reg, uint16_t &data);
    
    /** Write to a register (exposed for debugging reasons)
     *  @param reg - The register to be written
     */
    void writeRegister(uint16_t const reg);
    
    /** Allow the IC to run and collect user input
     */
    void enable(void);
   
    /** Stop the IC and put into low power mode
     */   
    void disable(void);
    
private:
    SPI         *_spi;
    DigitalOut  *_cs;
    DigitalOut  *_rst;
    InterruptIn *_dr;
    
    void drHandler(void);
};

#endif

