#include "mbed.h"
#include "rtos.h"

//Photointerrupter input pins
#define I1pin D2
#define I2pin D11
#define I3pin D12

//Incremental encoder input pins
#define CHA   D7
#define CHB   D8  

//Motor Drive output pins   //Mask in output byte
#define L1Lpin D4           //0x01
#define L1Hpin D5           //0x02
#define L2Lpin D3           //0x04
#define L2Hpin D6           //0x08
#define L3Lpin D9           //0x10
#define L3Hpin D10          //0x20

//Mapping from sequential drive states to motor phase outputs
/*
State   L1  L2  L3
0       H   -   L
1       -   H   L
2       L   H   -
3       L   -   H
4       -   L   H
5       H   L   -
6       -   -   -
7       -   -   -
*/
//Drive state to output table
const int8_t driveTable[] = {0x12,0x18,0x09,0x21,0x24,0x06,0x00,0x00};

//Mapping from interrupter inputs to sequential rotor states. 0x00 and 0x07 are not valid
const int8_t stateMap[] = {0x07,0x05,0x03,0x04,0x01,0x00,0x02,0x07};  
//const int8_t stateMap[] = {0x07,0x01,0x03,0x02,0x05,0x00,0x04,0x07}; //Alternative if phase order of input or drive is reversed

//Phase lead to make motor spin
const int8_t lead = -2;  //2 for forwards, -2 for backwards

//Status LED
DigitalOut led1(LED1);

//Define Photointerrupter inputs as Interrupts
InterruptIn I1 (I1pin);
InterruptIn I2 (I2pin);
InterruptIn I3 (I3pin);

//Globals
volatile int8_t intState = 0;
volatile int8_t intStateOld = 0;


//Motor Drive outputs
DigitalOut L1L(L1Lpin);
DigitalOut L1H(L1Hpin);
DigitalOut L2L(L2Lpin);
DigitalOut L2H(L2Hpin);
DigitalOut L3L(L3Lpin);
DigitalOut L3H(L3Hpin);

//Serial Comms
Serial pc(USBTX, USBRX);


void motorOut(int8_t driveState){
    
    //Lookup the output byte from the drive state.
    int8_t driveOut = driveTable[driveState & 0x07];
    
    //pc.printf("test: %x\n", driveState & 0x07);

    //Turn off first
    if (~driveOut & 0x01) L1L = 0;
    if (~driveOut & 0x02) L1H = 1;
    if (~driveOut & 0x04) L2L = 0;
    if (~driveOut & 0x08) L2H = 1;
    if (~driveOut & 0x10) L3L = 0;
    if (~driveOut & 0x20) L3H = 1;
    
    //Then turn on
    if (driveOut & 0x01) L1L = 1;
    if (driveOut & 0x02) L1H = 0;
    if (driveOut & 0x04) L2L = 1;
    if (driveOut & 0x08) L2H = 0;
    if (driveOut & 0x10) L3L = 1;
    if (driveOut & 0x20) L3H = 0;
    }
    
    //Convert photointerrupter inputs to a rotor state
void readRotorState(){
    intStateOld = intState;
    intState = stateMap[I1.read() + 2*I2.read() + 4*I3.read()];

    }

inline int8_t readRotorState1(){
    return stateMap[I1.read() + 2*I2.read() + 4*I3.read()];
    }


//Basic synchronisation routine    
int8_t motorHome() {
    
    //Put the motor in drive state 0 and wait for it to stabilise
    motorOut(0);
    wait(1.0);
    
    //Get the rotor state
    return readRotorState1();
}


int main() {
    
    //Attach ISR to interruptIns
    I1.rise(&readRotorState);
    I2.rise(&readRotorState);
    I3.rise(&readRotorState);


    int8_t orState = 0;    //Rotot offset at motor state 0
    //Run the motor synchronisation
    orState = motorHome();
    
    readRotorState();      //To ensure that the correct initial value for intState is used
    
    
    //orState is subtracted from future rotor state inputs to align rotor and motor states
    //Set the motor outputs accordingly to spin the motor
    while (1) {
        if (intState != intStateOld) {
            motorOut((intState-orState+lead+6)%6); //+6 to make sure the remainder is positive
        }
    }

}