/**
 * @file at24c1024b.h
 * @brief EEPROMのAT24C1024B用ライブラリ
 * @author 西川顕史
 * @date 2017/12/11（開始日）
 */
#ifndef __AT24C1024B_H__
#define __AT24C1024B_H__

#include "mbed.h"

static const int AT24C1024B_ADDRESS = 0xa0;

/**
* @brief AT24C1024Bの機能をまとめたクラス
* @details
*/
class AT24C1024B
{
public:
    //@{
    /**
    * @brief コンストラクタ
    * @param[in] sda i2c通信に使うsdaピンの指定
    * @param[in] scl i2c通信に使うsclピンの指定
    * @param[in] addr EEPROMのピンの状態によって変わるアドレス。省略で0(すべて開放もしもしくはGND)
    */

    AT24C1024B(PinName sda, PinName scl, int addr);
    AT24C1024B(PinName sda, PinName scl);
    AT24C1024B(I2C device, int addr);
    AT24C1024B(I2C device);
    //@}
    
    /**
    * @brief １バイト分のデータをEEPROMに書き込むための関数
    * @param[in] addr どのアドレスに書き込むか指定
    * @param[in] page どのページに書き込むか指定
    * @param[in] data 書き込むデータ
    * @return int エラーが発生したかを確認するためのもの
    * @details 正常終了したら0を返す。
    */
    int set_i2c_address(int addr, int page=0);
    int set_i2c_address(int A0, int A1, int page=0);
    
    /**
    * @brief データをEEPROMに書き込むための関数
    * @param[in] addr どのアドレスから書き込むか指定
    * @param[in] page どのページに書き込むか指定
    * @param[in] data 書き込むデータ
    * @param[in] *data 書き込むデータの配列
    * @param[in] length 書き込むデータの長さ
    * @return int エラーが発生したかを確認するためのもの
    * @details 内部で連続書き込みの関数を使用している。
    * 正常終了したら0を返す。
    * lengthが1未満だと-1を返す。
    * *dataは連続している必要がある
    */
    int write(int addr, int page, char data);
    int write(int addr, int page, char *data, int length);
    /**
    * @brief １バイト分データをEEPROMに書き込むための関数
    * @param[in] addr どのアドレスに書き込むか指定
    * @param[in] page どのページに書き込むか指定
    * @param[in] data 書き込むデータ
    * @return int エラーが発生したかを確認するためのもの
    * @details 正常終了したら0を返す。
    */
    int read(int addr, int page);
    int read(int addr, int page, char *data, int length);
private:
    PinName SDA;
    PinName SCL;
    I2C i2c;
    int ADDRESS;
};

#endif