#ifndef        _MCC0G42005A6W_H
#define        _MCC0G42005A6W_H

#include    <stdarg.h>
#include    "mbed.h"
#include    "MCC0G42005A6W.h"

/**
* LCD example
*
* @code
* #include    <stdarg.h>
* #include    "mbed.h"
* #include    "MCC0G42005A6W.h"
*
* // MCCOG42005A6W-BNMLWI
* // Alphanumeric LCD, 20 x 4, White on Blue, 3V to 5V, I2C, English, Japanese, Transmissive
* // Farnell nr. 2218946
* // example PCB  LCD I2C 4x20 https://circuitmaker.com/User/Details/Rob-Keij-4
*
*  LCD_COG lcd(SDA, SCL, "==LCD=Driver=V1.00==");  //  SDA, SCL
*
* int main()
* {
*  lcd.printf( 0, 0, "Hello world!" );
*  lcd.printf( 4, 1, "pi = %.6f", 3.14159265 );
*  lcd.printf( 2, 2, "This is Row %d",2 );
*  lcd.printf( 3, 3, "This is Row %d",3 );
*
*    while(1) {
*
*    }
* }
* @endcode
*/


class LCD_COG
{
public:

    /** Create a LCD instance which is connected to specified I2C pins with specified address
     *
     * @param I2C_sda I2C-bus SDA pin
     * @param I2C_scl I2C-bus SCL pin
     * @param init_message string to initialize the LCD
     */
    LCD_COG( PinName I2C_sda, PinName I2C_scl, const char *init_message = NULL );

    /** Create a LCD instance connected to specified I2C pins with specified address
     *
     * @param I2C object (instance)
     * @param init_message string to initialize the LCD
     */
    LCD_COG( I2C &i2c_, const char* init_message = NULL );

    /** Destructor
     */
    ~LCD_COG();

    /** Printf
     *
     *  printf function with line number.
     *  it can be used like
     *
     *  lcd.printf( 0, "Hello world!" );
     *  lcd.printf( 1, "pi = %.6f", 3.14159265 );
     *  lcd.printf( 2, "This is Row %d",2 );
     *  lcd.printf( 3, "This is Row %d",3 );
     *
     * @param line line# (0 for upper, 1 for lower)
     * @param format following parameters are compatible to stdout's printf
     */
    void printf( char line, const char *format, ... );

    /** Printf
     *
     *  printf function with X and Y character position on the LCD.
     *  it can be used like
     *
     *  lcd.printf( 0, 0, "Hello world!" );
     *  lcd.printf( 4, 1, "pi = %.6f", 3.14159265 );
     *  lcd.printf( 2, 2, "This is Row %d",2 );
     *  lcd.printf( 3, 3, "This is Row %d",3 );
     *
     * @param x X horizontal character position on the LCD
     * @param y Y vertical character position on the LCD
     * @param format following parameters are compatible to stdout's printf
     */
    void printf( char x, unsigned char y, const char *format, ... );

    /** Put character : "putc()"
     *
     * @param line line# (0 for upper, 1 for lower)
     * @param c character code
     */
    void putc( unsigned char line, char c );

    /** Put string : "puts()"
     *
     * @param line line# (0 for upper, 1 for lower)
     * @param s pointer to a string data
     */
    void puts( char line, const char *s );

    /** Put character into specified screen position
     *
     * @param c character code
     * @param x horizontal character position on the LCD
     * @param y vertical character position on the LCD
     */
    void putcxy( char c, unsigned char x, unsigned char y );

    /** Clear the LCD
     */
    void clear( void );

    /** Contrast adjustment
     *
     * @param contrast value (from 0x00 to 0x3E)
     */
    void contrast( char contrast );

    /** Put a custom character given as bitmap data
     *
     * @param c_code character code
     * @param cg pointer to bitmap data (array of 8 bytes)
     * @param x horizontal character position on the LCD
     * @param y vertical character position on the LCD
     */
    void put_custom_char( char c_code, const char *cg, char x, char y );

    /** Set CGRAM (set custom bitmap as a character)
     *
     * @param c_code character code
     * @param cg pointer to bitmap data (array of 8 bytes)
     */
    void set_CGRAM( char char_code, const char* cg );

    /** Set CGRAM (set custom bitmap as a character)
     *
     * @param c_code character code
     * @param v bitmap data (5 bit pattern in this variable are copied to all row of a character bitmap)
     */
    void set_CGRAM( char char_code, char v );


    /** Set number of characters in a line
     *
     * @param ch number of charactors in a line
     */
    void setCharsInLine( char ch )
    {
        charsInLine = ch;
    };

private:

    typedef enum {

        SLAVEADRESS                  = 0x78,
        LINES                        = 4,
        LCD_HOME_L1                  = 0x80,
        LINE1                        = 0x00,
        LINE2                        = LINE1+0x20,
        LINE3                        = LINE1+0x40,
        LINE4                        = LINE1+0x60,

        DISPLAY_ON                   = 0x04,
        DISPLAY_OFF                  = 0x03,
        CURSOR_ON                    = 0x02,
        CURSOR_OFF                   = 0x05,
        BLINK_ON                     = 0x01,
        BLINK_OFF                    = 0x06,

        TOPVIEW                      = 0x05,
        BOTTOMVIEW                   = 0x06,

        ROMA                         = 0x00,
        ROMB                         = 0x04,
        ROMC                         = 0x0C,

        Comm_FunctionSet_Normal      = 0x38,
        Comm_FunctionSet_Extended    = 0x39,
        Comm_InternalOscFrequency    = 0x14,
        Comm_DisplayOnOff            = 0x0C,
        Comm_ClearDisplay            = 0x01,
        Comm_ReturnHome              = 0x02,
        Comm_ContrastSet             = 0x70,
        Comm_PwrIconContrast         = 0x5C,
        Comm_FollowerCtrl            = 0x60,
        Comm_EntryModeSet            = 0x04,
        Comm_SetCGRAM                = 0x40


//Instruction     IS RE RS R/W   DB7 DB6 DB5 DB4 DB3 DB2 DB1 DB0 Description
//Clear display           XX00   0   0   0   0   0   0   0   1   Write "20H" to DDRAM. and set DDRAM address to "00H" from AC.
//Comm_ClearDisplay = 0x01,
//
//Return home             X000   0   0   0   0   0   0   1   X   Set DDRAM address to "00H" from AC and return cursor to its original position if shifted. The contents of DDRAM are not changed.
//Comm_ReturnHome   = 0x02,
//
//Power down mode         X100   0   0   0   0   0   0   1   PD  Set power down mode bit.  PD = "1": power down mode set,  PD = "0": power down mode disable (POR)
//
//Entry mode set          X000   0   0   0   0   0   1   I/D S   Assign cursor/ blink moving direction with DDRAM address I/D = "1": cursor/ blink moves to right
//                                                               and DDRAM address is increased by 1 (POR)
//                                                               I/D = "0": cursor/ blink moves to left and DDRAM address is decreased by 1
//                                                               Assign display shift with DDRAM address S = "1": make display shift of the enabled lines by the DS4 to DS1 bits in the shift enable instruction.
//                                                               Left/right direction depends on I/D bit selection. S = "0": display shift disable (POR)
//
//                        X100   0   0   0   0   0   1   BDC BDS Segment bi-direction function.
//                                                               BDS = "0": Seg100 -> Seg1,
//                                                               BDS = "1": Seg1 -> Seg100.
//                                                               Segment bi-direction function.
//                                                               BDC = "0": Com32 -> Com1
//                                                               BDC = "1": Com1 -> Com32
//
//Display On/Off control  X000   0   0   0   0   1   D   C   B   Set display/cursor/blink on/off
//Comm_DisplayOnOff = 0x0C, // 0000 1100
//                                                               D = "1": display on,
//                                                               D = "0": display off (POR),
//                                                               C = "1": cursor on,
//                                                               C = "0": cursor off (POR),
//                                                               B = "1": blink on,
//                                                               B = "0": blink off (POR).
//
//Extended function set   X100   0   0   0   0   1   FW  B/W NW  Assign font width, black/white inverting of cursor, and 4-line display mode control bit.
//                                                               FW = "1": 6-dot font width,
//                                                               FW = "0": 5-dot font width (POR),
//                                                               B/W = "1": black/white inverting of cursor enable,
//                                                               B/W = "0": black/white inverting of cursor disable (POR)
//                                                               NW = "1": 3-line or 4-line display mode,
//                                                               NW = "0": 1-line or 2-line display mode
//
//Cursor or display shift 0000   0   0   0   1   S/C  R/L x  x   Set cursor moving and display shift control bit, and the direction, without changing DDRAM data.
//                                                               S/C = "1": display shift,
//                                                               S/C = "0": cursor shift,
//                                                               R/L = "1": shift to right,
//                                                               R/L = "0": shift to left.
//
//Double height(4-line)/  0100   0   0   0   1   UD2 UD1 BS1 DH’ UD2~1: Assign different doubt height format (POR=11)
//Bias/Display-dotshift                                          BS1:BS0 = “00”: 1/5 bias (POR)
//                                                               BS1:BS0 = “01”: 1/4 bias
//                                                               BS1:BS0 = “10”: 1/7 bias
//                                                               BS1:BS0 = “11”: 1/6 bias
//                                                               DH’ = "1": display shift enable
//                                                               DH’ = "0": dot scroll enable (POR)


//Internal OSC frequency  1000   0   0   0    1  BS0 F2  F1  F0  F2~0: adjust internal OSC frequency for FE frequency (POR: 011)
//Comm_InternalOscFrequency   = 0x14,
//
//Shiftenable             1100   0   0   0    1  DS4 DS3 DS2 DS1 (when DH’ = "1") POR DS4~1=1111 Determine the line for display shift.
//                                                               DS1 = "1/0": 1st line display shift enable/disable
//                                                               DS2 = "1/0": 2nd line display shift enable/disable
//                                                               DS3 = "1/0": 3rd line display shift enable/disable
//                                                               DS4 = "1/0": 4th line display shift enable/disable.
//Scroll enable           1100   0   0   0    1  HS4 HS3 HS2 HS1 (when DH’ = "0") POR HS4~1=1111 Determine the line for horizontal smooth scroll.
//                                                               HS1 = "1/0": 1st line dot scroll enable/disable
//                                                               HS2 = "1/0": 2nd line dot scroll enable/disable
//                                                               HS3 = "1/0": 3rd line dot scroll enable/disable
//                                                               HS4 = "1/0": 4th line dot scroll enable/disable.
//
//Function set            X000   0   0   1    DL  N  DH  RE(0) IS Set interface data length  DL = "1": 8-bit (POR),
//                                                                                           DL = "0": 4-bit
//                                                                Numbers of display line     N = "1": 2-line (NW=0)/ 4-line(NW=1),
//                                                                                            N = "0": 1-line (NW=0)/ 3-line(NW=1)
//                                                                Extension register, RE("0") at this instruction, RE must be "0".
//                                                                Shift/scroll enable DH = “ 1/0”: Double height font control for 2-line mode enable/ disable (POR=0)
//                                                                IS Special registers enable bit at this moment, must be "0".


//Comm_FunctionSet_Normal    = 0x3a, // 0011 1010
//Comm_FunctionSet_Normal    = 0x38, // 0011 1000


//
//
//                        X100   0   0   1    DL  N BE RE(1) REV Set DL, N, RE("1") CGRAM/SEGRAM blink enable BE = " 1/0": CGRAM/SEGRAM blink enable/disable (POR=0)
//                                                               Reverse bit REV = "1": reverse display, REV = "0": normal display (POR).
//
//set CGRAM address       0000   0   1   AC5  AC4 AC3 AC2 AC1 AC0 Set CGRAM address in address counter. (POR=00 0000)

//set SEGRAM address      1000   0   1   0    0   AC3 AC2 AC1 AC0 Set SEGRAM address in address counter. (POR=0000)
//
//Power/
//Icon control/
//Contrast set            1000   0   1   0    1   Ion Bon C5 C4   Ion = “1/0”: ICON (SEGRAM) display on/off (POR=0)
//                                                                Bon = “1/0”: set booster and regulator circuit on/off (POR=0)
//                                                                C5, C4: Contrast set for internal follower mode (POR=10)
//
//Follower Control        1000   0   1   1    0  Don Rab2 Rab1 Rab0 Don: Set divider circuit on/ off (POR=0)
//                                                                Rab2~0: Select Amplifier internal resistor ratio (POR=010)
//
//Contrast Set            1000   0   1   1    1  C3  C2   C1   C0 C3~0: Contrast set for internal follower mode (POR=0000)
//
//set DDRAM address       X000   1  AC6 AC5  AC4 AC3 AC2 AC1  AC0 Set DDRAM address in address counter. (POR=000 0000)
//
//set scroll quantity     X100   1  X   SQ5  SQ4 SQ3 SQ2 SQ1  SQ0 Set the quantity of horizontal dot scroll. (POR=00 0000)
//
//Read busy flag and                                               Can be known whether during internal operation or not by reading BF.
//address/part ID         XX01  BF  AC6  AC5  AC4  AC3 AC2 AC1 AC0 The contents of address counter or the part ID can also be read.
//                                   /    /    /    /   /   /   /  When it is read the first time, the address counter can be read.
//                                  ID6  ID5  ID4  ID3 ID2 ID1 ID0 When it is read the second time, the part ID can be read.
//
//                                                                  BF = "1": busy state
//                                                                  BF = "0": ready state
//
//write data              XX10  D7  D6   D5 D4 D3 D2 D1 D0          Write data into internal RAM (DDRAM / CGRAM / SEGRAM).
//read data               XX11  D7  D6   D5 D4 D3 D2 D1 D0          Read data from internal RAM (DDRAM / CGRAM / SEGRAM).
//
//


///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//                                               Extended Instruction Set
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//Temperature             X100  0   1    1   1  0  1  1  0          Set Temperature Coefficient TC2~0:
//Coefficient                                                       000: Reserved
//Control                                                           001: Reserved
//                                                                  010: -0.05%/ﾟC (POR)
//Temperature             XX10  0   0    0   0  0 TC2 TC1 TC0       011: Reserved
//Coefficient                                                       100: -0.10%/ﾟC
//Control                                                           101: Reserved
//Settings                                                          110: -0.15%/ﾟC
//                                                                  111: -0.20%/ﾟC
//
//ROM Selection           X100  0   1    1   1    0   0   1 0       Writing data into ROM selection
//ROM Selection Settings  XX10  0   0    0   0  ROM2 ROM1 0 0       register enables the selection of
//                                                                  ROM A, B or C.
//                                                                  ROM2~1:
//                                                                  00: ROMA
//                                                                  01: ROMB
//                                                                  10: ROMC
//                                                                  11: Invalid

    } _commands;

    typedef enum {
        MaxCharsInALine            = 0x14, //    buffer depth for one line (no scroll function used)
        COMMAND                    = 0x00,
        DATA                       = 0x40
    } _constants;


    char    curs[LINES];                          // Character position on line
    void    init( const char* init_message );
    void    clear_rest_of_line( unsigned char line );
    int     lcd_write( char first, char second );
    int     lcd_command( char command );
    int     lcd_data( char data );
    I2C     *i2c_p;
    I2C     &i2c;
    char    i2c_addr;
    char    charsInLine;

};


#endif










