/* This program demonstrates how to use the board to play a reaction time game.
 * The winner and his reaction time are displayed in the console window and transmitted via Sigfox.
 * Open a serial connection to the board to see extra info and instructions.
 */
 
#include "mbed.h"

#include <stdio.h>
#include <stdlib.h>

#define SER_BUFFER_SIZE 32

/* The 4 onboard LEDs */
DigitalOut LED_0 (PB_6);
DigitalOut LED_1 (PA_7);
DigitalOut LED_2 (PA_6);
DigitalOut LED_3 (PA_5);

/* The 2 user buttons */
InterruptIn SW1(PA_8);
InterruptIn SW2(PB_10);

/* Function prototypes */
void sw1interrupt();
void sw2interrupt();
void sertmout();
bool modem_command_check_ok(char * command);
void modem_setup();

bool ser_timeout = false; 

/* Serial port over USB */
Serial pc(USBTX, USBRX);

/* Serial connection to sigfox modem */
Serial modem(PA_9, PA_10);

/* User variables */
uint16_t randomTime;
uint16_t reactionTime;
uint8_t winner;

bool play = false;
bool player1 = false;
bool player2 = false;
bool busyPlaying =  false;

int main() {   
    
    time_t t;
    
    /* Setup TD120x */
    wait(3);
    modem_setup();
    
    /* Turn off all LED */
    LED_0 = 1;
    LED_1 = 1;
    LED_2 = 1;
    LED_3 = 1;
    
    /* Setup button interrupts */    
    SW1.fall(&sw1interrupt);
    SW2.fall(&sw2interrupt); 
    
    /* Intializes random number generator */
    srand((unsigned) time(&t));
    
    /* Push a button to start the first game */    
    pc.printf("\nPress a button to start the reaction time game...\n\r");
    
    while(1) {          
        
        /* When waiting to start a new game, show a loop on the leds */
        while(play == false){            
            if(play == false){  // Check every time, otherwise after interrupt, loop would be finished first
                LED_0 = 1;LED_1 = 1;LED_2 = 1;LED_3 = 0;
                wait_ms(100);
            }
            if(play == false){
                LED_0 = 1;LED_1 = 1;LED_2 = 0;LED_3 = 1;
                wait_ms(100); 
            }
            if(play == false){
                LED_0 = 1;LED_1 = 0;LED_2 = 1;LED_3 = 1;
                wait_ms(100); 
            }
            if(play == false){
                LED_0 = 0;LED_1 = 1;LED_2 = 1;LED_3 = 1;
                wait_ms(100);   
            }
            if(play == false){
                LED_0 = 1;LED_1 = 0;LED_2 = 1;LED_3 = 1;
                wait_ms(100);      
            }
            if(play == false){
                LED_0 = 1;LED_1 = 1;LED_2 = 0;LED_3 = 1;
                wait_ms(100); 
            }                                 
        }  
        
        /* After the 4 leds flash on, calculate the reaction time */
        while(busyPlaying == true && player1==false && player2 == false){
            reactionTime++;
            wait_ms(1);
        }
        
        /* When player 1 has won */       
        if(player1 == true){
            pc.printf("\nPlayer 1 was the fastest. His time was: %d ms\n\r", reactionTime);            
            LED_0 = 1;
            LED_1 = 1;
            LED_2 = 1;
            LED_3 = 1;  
            winner = 1;          
            char command[SER_BUFFER_SIZE];
            sprintf(command, "AT$SF=06%04x%04x,2,0\n", (int) winner, (int) reactionTime );
            pc.printf("Sending winner = %i and reaction time =  %i ms over Sigfox\n", winner, reactionTime);
            pc.printf("using modem command: %s\n", command);
            modem_command_check_ok(command);
            player1 = false;
            player2= false;
            busyPlaying = false;            
            play = false;
            reactionTime = 0;
            pc.printf("\nPress a button to start a new game...\n\r");
        }
        
        /* When player 2 has won */
        if(player2 == true){
            pc.printf("\nPlayer 2 was the fastest. His time was: %d ms\n\r", reactionTime);            
            LED_0 = 1;
            LED_1 = 1;
            LED_2 = 1;
            LED_3 = 1; 
            winner = 2;           
            char command[SER_BUFFER_SIZE];
            sprintf(command, "AT$SF=06%04x%04x,2,0\n", (int) winner, (int) reactionTime );
            pc.printf("Sending winner = %i and reaction time =  %i ms over Sigfox\n", winner, reactionTime);
            pc.printf("using modem command: %s\n", command);
            modem_command_check_ok(command);
            player1 = false;
            player2= false;
            busyPlaying = false;            
            play = false;
            reactionTime = 0;
            pc.printf("\nPress a button to start a new game...\n\r");
        }        
        
        /* Check if a new game has been started, when a game has been started then generate a random time */
        if(play == true){
            pc.printf("\nGame has started, push your button as quick as possible when leds go on!\n\r");
            randomTime =   (rand() % 5000)+1;   //random wait max 5 seconds
            wait_ms( randomTime + 1000 ); // wait for random time + 1 second
            busyPlaying =  true;
            LED_0 = 0;
            LED_1 = 0;
            LED_2 = 0;
            LED_3 = 0;
        } 
        
        
    }
}

void modem_setup()
{
    /* Reset to factory defaults */
    if(modem_command_check_ok("AT&F")) 
    {
        pc.printf("Factory reset succesfull\r\n");
    }
    else 
    {
        pc.printf("Factory reset TD120x failed\r\n");
    }
    /* Disable local echo */
    modem.printf("ATE0\n");
    if(modem_command_check_ok("ATE0")) 
    {
        pc.printf("Local echo disabled\r\n");
    }
    /* Write to mem */
    if(modem_command_check_ok("AT&W")) 
    {
        pc.printf("Settings saved!\r\n");
    }    
}

bool modem_command_check_ok(char * command)
{
    /* First clear serial data buffers */
    while(modem.readable()) modem.getc();
    /* Timeout for response of the modem */
    Timeout tmout;
    ser_timeout = false;
    /* Buffer for incoming data */
    char responsebuffer[6];
    /* Flag to set when we get 'OK' response */
    bool ok = false;
    bool error = false;
    /* Print command to TD120x */
    modem.printf(command);
    /* Newline to activate command */
    modem.printf("\n");
    /* Wait untill serial feedback, min 7 seconds before timeout */
    tmout.attach(&sertmout, 7.0);
    while(!modem.readable()&& ser_timeout == false);
    while(!ok && !ser_timeout && !error) 
    {
        if(modem.readable()) 
        {
            for(int i = 0; i < 5; i++)
            {
                responsebuffer[i] = responsebuffer[i+1];
            }
            responsebuffer[5] = modem.getc();
            if(responsebuffer[0] == '\r' && responsebuffer[1] == '\n' && responsebuffer[2] == 'O' && responsebuffer[3] == 'K' && responsebuffer[4] == '\r' && responsebuffer[5] == '\n' ) 
            {
                ok = true;
            }
            else if(responsebuffer[0] == '\r' && responsebuffer[1] == '\n' && responsebuffer[2] == 'E' && responsebuffer[3] == 'R' && responsebuffer[4] == 'R' && responsebuffer[5] == 'O' ) 
            {
                error = true;
            }
        }
    }
    tmout.detach();
    return ok;
}

/* Button 1 ISR */
void sw1interrupt()
{    
        if(play == false && busyPlaying == false){          // If not already playing... 
         play = true;
         LED_0 = 1;
         LED_1 = 1;
         LED_2 = 1;
         LED_3 = 1;
        }
        if(busyPlaying == true && player2 != true){         // When playing and there is no winner yet...
            player1 = true;
        }   
         
}

/* Button 2 ISR */
void sw2interrupt()
{    
        if(play == false && busyPlaying == false){          // If not already playing... 
         play = true;   
         LED_0 = 1;
         LED_1 = 1;
         LED_2 = 1;
         LED_3 = 1; 
        }
        if(busyPlaying == true && player1 != true){         // When playing and there is no winner yet...
            player2 = true;
        }       
    
}

/* ISR for serial timeout */
void sertmout()
{
    ser_timeout = true;
}
