/* This program demonstrates how to use the board to simulate a dual dice throw.
 * The result is displayed in binary on the leds and transmitted via Sigfox.
 * Open a serial console to the board to see extra info.
 */
 
#include "mbed.h"

#include <stdio.h>
#include <stdlib.h>

#define SER_BUFFER_SIZE 32

/* The 4 onboard LEDs */
DigitalOut LED_0 (PB_6);
DigitalOut LED_1 (PA_7);
DigitalOut LED_2 (PA_6);
DigitalOut LED_3 (PA_5);

/* The 2 user buttons */
InterruptIn SW1(PA_8);
InterruptIn SW2(PB_10);



/* Function prototypes */
void sw1interrupt();
void sw2interrupt();
void sertmout();
bool modem_command_check_ok(char * command);
void modem_setup();

bool ser_timeout = false; 


/* Serial port over USB */
Serial pc(USBTX, USBRX);

/* Serial connection to sigfox modem */
Serial modem(PA_9, PA_10);

bool dice1 = false;
bool dice2 = false;
uint8_t dice1val;
uint8_t dice2val;
uint8_t number;


int main() {   
    
    time_t t;
    
    /* Setup TD120x */
    wait(3);
    modem_setup();
    
    /* Turn off all LED */
    LED_0 = 1;
    LED_1 = 1;
    LED_2 = 1;
    LED_3 = 1;
    
    /* Setup button interrupts */    
    SW1.fall(&sw1interrupt);
    SW2.fall(&sw2interrupt); 
    
    /* Intializes random number generator */
    srand((unsigned) time(&t));     
    
    while(1) { 
        
        if(dice1 == true && dice2 == true){
            number = dice1val + dice2val;
            pc.printf("\nThe number is: %d\n\r", number);            
            
            switch(number) {
                case 2  :
                    LED_0 = 1;
                    LED_1 = 1;
                    LED_2 = 0;
                    LED_3 = 1;
                    break; 
                case 3  :
                    LED_0 = 1;
                    LED_1 = 1;
                    LED_2 = 0;
                    LED_3 = 0;
                    break;
                case 4  :
                    LED_0 = 1;
                    LED_1 = 0;
                    LED_2 = 1;
                    LED_3 = 1;
                    break; 
                case 5  :
                    LED_0 = 1;
                    LED_1 = 0;
                    LED_2 = 1;
                    LED_3 = 0;
                    break; 
                case 6  :
                    LED_0 = 1;
                    LED_1 = 0;
                    LED_2 = 0;
                    LED_3 = 1;
                    break; 
                case 7 :
                    LED_0 = 1;
                    LED_1 = 0;
                    LED_2 = 0;
                    LED_3 = 0;
                    break; 
                case 8  :
                    LED_0 = 0;
                    LED_1 = 1;
                    LED_2 = 1;
                    LED_3 = 1;
                    break;
                case 9  :
                    LED_0 = 0;
                    LED_1 = 1;
                    LED_2 = 1;
                    LED_3 = 0;
                    break; 
                case 10 :
                    LED_0 = 0;
                    LED_1 = 1;
                    LED_2 = 0;
                    LED_3 = 1;
                    break; 
                case 11  :
                    LED_0 = 0;
                    LED_1 = 1;
                    LED_2 = 0;
                    LED_3 = 0;
                    break;  
                case 12  :
                    LED_0 = 0;
                    LED_1 = 0;
                    LED_2 = 1;
                    LED_3 = 1;
                    break;                
                default : 
                    pc.printf("\nImpossible number!!\n\r");
                    LED_0 = 1;
                    LED_1 = 1;
                    LED_2 = 1;
                    LED_3 = 1;
            }
            char command[SER_BUFFER_SIZE];
            sprintf(command, "AT$SF=04%04x,2,0\n", (int) number );
            pc.printf("Sending thrown number =  %i over Sigfox using modem command: %s\n", number , command);
            modem_command_check_ok(command);
            //wait_ms(5000);
            LED_0 = 1;
            LED_1 = 1;
            LED_2 = 1;
            LED_3 = 1;
            dice1 = false;
            dice2 = false;
        }        
        
    }
}
void modem_setup()
{
    /* Reset to factory defaults */
    if(modem_command_check_ok("AT&F")) 
    {
        pc.printf("Factory reset succesfull\r\n");
    }
    else 
    {
        pc.printf("Factory reset TD120x failed\r\n");
    }
    /* Disable local echo */
    modem.printf("ATE0\n");
    if(modem_command_check_ok("ATE0")) 
    {
        pc.printf("Local echo disabled\r\n");
    }
    /* Write to mem */
    if(modem_command_check_ok("AT&W")) 
    {
        pc.printf("Settings saved!\r\n");
    }    
}

bool modem_command_check_ok(char * command)
{
    /* first clear serial data buffers */
    while(modem.readable()) modem.getc();
    /* Timeout for response of the modem */
    Timeout tmout;
    ser_timeout = false;
    /* Buffer for incoming data */
    char responsebuffer[6];
    /* Flag to set when we get 'OK' response */
    bool ok = false;
    bool error = false;
    /* Print command to TD120x */
    modem.printf(command);
    /* Newline to activate command */
    modem.printf("\n");
    /* Wait untill serial feedback, min 7 seconds before timeout */
    tmout.attach(&sertmout, 7.0);
    while(!modem.readable()&& ser_timeout == false);
    while(!ok && !ser_timeout && !error) 
    {
        if(modem.readable()) 
        {
            for(int i = 0; i < 5; i++)
            {
                responsebuffer[i] = responsebuffer[i+1];
            }
            responsebuffer[5] = modem.getc();
            if(responsebuffer[0] == '\r' && responsebuffer[1] == '\n' && responsebuffer[2] == 'O' && responsebuffer[3] == 'K' && responsebuffer[4] == '\r' && responsebuffer[5] == '\n' ) 
            {
                ok = true;
            }
            else if(responsebuffer[0] == '\r' && responsebuffer[1] == '\n' && responsebuffer[2] == 'E' && responsebuffer[3] == 'R' && responsebuffer[4] == 'R' && responsebuffer[5] == 'O' ) 
            {
                error = true;
            }
        }
    }
    tmout.detach();
    return ok;
}
/* Button 1 ISR */
void sw1interrupt()
{
    if(dice1 == false){
        pc.printf("\nDice 1 has been thrown\n\r");
        dice1 =  true; 
        dice1val = (rand() % 6)+1; 
    }
    else{
        pc.printf("\nDice 1 has already been thrown\n\r");
    }
         
}

/* Button 2 ISR */
void sw2interrupt()
{
    if(dice2 == false){
        pc.printf("\nDice 2 has been thrown\n\r");
        dice2 =  true; 
        dice2val = (rand() % 6)+1; 
    }
    else{
        pc.printf("\nDice 2 has already been thrown\n\r");
    }
}

/* ISR for serial timeout */
void sertmout()
{
    ser_timeout = true;
}



