#include "mbed.h"
#include "math.h"
#include "LinearTempSensor.h"

/* The 4 onboard LEDs */
DigitalOut LED_0 (PB_6);
DigitalOut LED_1 (PA_7);
DigitalOut LED_2 (PA_6);
DigitalOut LED_3 (PA_5);

/* The 2 user buttons */
InterruptIn SW1(PA_8);
InterruptIn SW2(PB_10);

/*Temperature sensor */
LinearTempSensor sensor(PA_0, 5);

/* Function prototypes */
void sw1interrupt();
void sw2interrupt();
void beat();
void sertmout();
bool modem_command_check_ok(char * command);
void modem_setup();
void txData(uint8_t btn);
bool ser_timeout = false;

Ticker heartbeat;

/* Serial port over USB */
Serial pc(USBTX, USBRX);

/* Serial connection to sigfox modem */
Serial modem(PA_9, PA_10);


int main()
{

    /* Setup TD120x */
    wait(3);
    modem_setup();
    /* Test temperature sensor */
    float vOut = sensor.Sense();
    pc.printf("\n\rMCP9700 reading:  Vout: %.2f mV", vOut);

    /* Turn off all LED */
    LED_0 = 1;
    LED_1 = 1;
    LED_2 = 1;
    LED_3 = 1;

    /* Blinking LED */
    heartbeat.attach(&beat, 0.5);

    /* Setup button interrupts */
    SW2.fall(&sw2interrupt);
    SW1.fall(&sw1interrupt);

    while(1) {
        if(pc.readable()) {
            modem.putc(pc.getc());
        }
        if(modem.readable()) {
            pc.putc(modem.getc());
        }
    }
}

void beat()
{
    LED_0 = !LED_0;
}

/* Button 1 ISR */
void sw1interrupt()
{
    pc.printf("\n\rButton 1 pressed\n\r");
    LED_1 = 0;
    txData(1);
}

/* Button 2 ISR */
void sw2interrupt()
{
    pc.printf("\n\rButton 2 pressed\n\r");
    LED_2 = 0;
    txData(2);
}


/* TX data over Sigfox */
void txData (uint8_t btn)
{
    float    tAvg;
    char     command[32];
    sensor.Sense();
    tAvg = sensor.GetAverageTemp();
    char temperature[6] ="";
    sprintf(temperature, "%3.1f", tAvg);
    for(int i = 0; i < 5; i++)
        if(temperature[i]==0) temperature[i] = ' ';
    sprintf(command, "AT$SF=0142544e%x20%x%x%x%x%x43,2,0\n", btn+48, temperature[0],temperature[1],temperature[2],temperature[3],temperature[4]);
    pc.printf("Sending pressed button %d and temperature %s C over Sigfox.\n", btn, temperature);
    pc.printf("using modem command: %s", command);
    modem_command_check_ok(command);
    LED_1 = 1;
    LED_2 = 1;
}

void modem_setup()
{
    /* Reset to factory defaults */
    if(modem_command_check_ok("AT&F")) {
        pc.printf("Factory reset succesfull\r\n");
    } else {
        pc.printf("Factory reset TD120x failed\r\n");
    }
    /* Disable local echo */
    modem.printf("ATE0\n");
    if(modem_command_check_ok("ATE0")) {
        pc.printf("Local echo disabled\r\n");
    }
    /* Write to mem */
    if(modem_command_check_ok("AT&W")) {
        pc.printf("Settings saved!\r\n");
    }
}

/* ISR for serial timeout */
void sertmout()
{
    ser_timeout = true;
}

bool modem_command_check_ok(char * command)
{
    /* first clear serial data buffers */
    while(modem.readable()) modem.getc();
    /* Timeout for response of the modem */
    Timeout tmout;
    ser_timeout = false;
    // check if ok or error is received
    char * readyString = "OK";
    char * readyStringPtr;
    readyStringPtr = readyString;
    char * errorString = "ERROR";
    char * errorStringPtr;
    errorStringPtr = errorString;
    /* Flag to set when we get 'OK' response */
    bool ok = false;
    bool error = false;
    /* Print command to TD120x */
    modem.printf(command);
    /* Newline to activate command */
    modem.printf("\n");
    /* Wait untill serial feedback, max 10 seconds before timeout */
    tmout.attach(&sertmout, 10.0);
    int c;
    while (!ser_timeout  && !ok && !error) {
        if(modem.readable()) {
            c = modem.getc();
            if ( (char)c == *readyStringPtr ) readyStringPtr++;
            else readyStringPtr = readyString;
            if ( *readyStringPtr == 0 ) {
                ok = true;
            }
            if ( (char)c == *errorStringPtr ) errorStringPtr++;
            else errorStringPtr = errorString;
            if ( *errorStringPtr == 0 ) {
                error = true;
            }
        }
    }
    tmout.detach();
    if(ser_timeout) return false;
    else return ok;
}