// ColourSensor.cpp

#include "ColourSensor_H.h"
#include "mbed.h"

ColourSensor::ColourSensor(PinName If, PinName Which, PinName Out):
    inIf(If), inWhich(Which), solenoid(Out) {
        // Class constructor for colour sensor & solenoid
        // Sets initial conditions of operation
        
        initialConditions();
}

void ColourSensor::initialConditions() {
    // set initial conditions & variables
    //printf("Setting Initial Conditions\n\r");
    detectedIf = 0;
    detectedIfOld = 0;
    newDetection = 0;
    diskHave = 0;
    
    solenoidState = 0;
    
    toggleA = 0;
    toggleB = 0;
    toggleConst = 1;
    
    solenoidSet(0);
}

void ColourSensor::solenoidSet(bool state) {
    // Turns solenoid on/off, sets solenoid's pull strength
    // state: pass 1 or 0 for solenoid ON or OFF respectively
    
    if (state) {
        solenoid.write(1);
        //printf("Solenoid has been turn ON\n\r");
    }
    
    else {
        solenoid.write(0);
        //printf("Solenoid has been turned OFF\n\r");
    }
    
    solenoidState = state;
}

void ColourSensor::readIf() {
    // Interrupt function (software): reads in colour detection state
    
    // Update variables
    detectedIfOld = detectedIf;
    newDetection = 0;
    
    //printf("Reading colour detection state\n\r");
    
    detectedIf = !inIf.read();
    
    if ((detectedIf && !detectedIfOld)) {
        // if colour is newly detected
        newDetection = 1;
        toggleA = 1;
        //printf("Colour newly detected\n\r");
    }
}

void ColourSensor::readWhich() {
    //Interrupt function (software): reads in which colour is detected
    
    detectedWhich = !inWhich.read();
    
    //printf("Colour detected; code: %d\n\r", detectedWhich);

    toggleB = 1;
}

void ColourSensor::makeColourActive() {
    // Interrupt function (software): reactivates colour processing 
    toggleConst = 1;
    
    //printf("Colour processing Activated\n\r");
}

void ColourSensor::process() {
    // Processes change of state in colour detection
    //printf("Colour process() called\n\r");
    
    if (!diskHave && newDetection) {
        // If: No disk & colour newly detected, turn on solenoid
        //printf("Colour detected; collecting disk\n\r");
        
        solenoidSet(1);
        diskHave = 1;
        diskColour = detectedWhich;
    } 
    
    else if ((diskHave && newDetection) && (detectedWhich == diskColour)) {
        // If: Have disk & colour newly detected & disk colour is same as detected colour, turn off solenoid
        //printf("Correct colour detected; depositing disk\n\r");
        solenoidSet(0);
        diskHave = 0;
    }
    
    else if ((diskHave && newDetection) && (detectedWhich != diskColour)) {
        // If: Have disk & colour newly detected & disk colour is NOT same as detected colour, update variables
        //printf("Wrong colour detected; keeping disk\n\r");
    }
    
    else {
        //printf("Cool (y)\n\r");
    }
    
    // Temporarily disable colour processing
    //printf("Disabling colour processing\n\r");
    toggleConst = 0;
}