#include "mbed.h"
#include "TSIHW.h"

/* TSI constructor
Parameters: the values of the fields REFCHRG, EXTCHRG, DVOLT, PS, and NSCN from TSI0_GENCS register
Not using interrupts, capacitive sensing (non-noise detection), not configured to work in STOP modes
*/
TSI::TSI(char rchg, char echg, char dvolt, char ps, char nscn) {
    // The first version is preconfigured for non-noise detection, no interrupts, not running on stop modes, software trigger
    SIM->SCGC5 |= 0x00000E20; // clock gate for TSI and PORTS A, B, and C
    TSI0->GENCS = 0xC0000080 | ((rchg & 0x07) << 21) | ((echg & 0x07) << 16) | ((dvolt & 0x03) << 19) | ((ps & 0x07) << 13) | ((nscn & 0x1F) << 8);
}

/* TSI destructor */
TSI::~TSI() { }

/* Function to configure a pin to work with the corresponding channel (passed as the single parameter) */
void TSI::ActivateChannel(char ch) {
    // reads channel number and sets the MUX of the corresponding pin to ALT0 function
    switch(ch) {
        case 0: PORTB->PCR[0] &= 0xFFFFF8FF; break;
        case 1: PORTA->PCR[0] &= 0xFFFFF8FF; break;
        case 2: PORTA->PCR[1] &= 0xFFFFF8FF; break;
        case 3: PORTA->PCR[2] &= 0xFFFFF8FF; break;
        case 4: PORTA->PCR[3] &= 0xFFFFF8FF; break;
        case 5: PORTA->PCR[4] &= 0xFFFFF8FF; break;
        case 6: PORTB->PCR[1] &= 0xFFFFF8FF; break;
        case 7: PORTB->PCR[2] &= 0xFFFFF8FF; break;
        case 8: PORTB->PCR[3] &= 0xFFFFF8FF; break;
        case 9: PORTB->PCR[16] &= 0xFFFFF8FF; break;
        case 10: PORTB->PCR[17] &= 0xFFFFF8FF; break;
        case 11: PORTB->PCR[18] &= 0xFFFFF8FF; break;
        case 12: PORTB->PCR[19] &= 0xFFFFF8FF; break;
        case 13: PORTC->PCR[0] &= 0xFFFFF8FF; break;
        case 14: PORTC->PCR[1] &= 0xFFFFF8FF; break;
        case 15: PORTC->PCR[2] &= 0xFFFFF8FF; break;
        default: error("PinName provided to TSI::ActivateChannel() does not correspond to any known TSI channel.");
    }
}

// Function to trigger the reading of a given channel
void TSI::Start(char ch) {
    //writes 1 to the software trigger bit, defining the channel number in the respective bits
    TSI0->GENCS |= 0x00000004; // clears EOSF
    TSI0->DATA = 0x00400000 | (ch << 28);
}

// Function to read scan result; returns zero if not finished
uint16_t TSI::Read() {
    uint16_t aux;
    
    if(!(TSI0->GENCS & 0x00000004)) {
        return 0;
    } else {
        aux = TSI0->DATA & 0x0000FFFF;
        TSI0->GENCS |= 0x00000004; // clears EOSF
        return aux;
    }
}
