/**
@file N5110.cpp

@brief Member functions implementations

*/

#include "N5110.h"
#include "mbed.h"
#include "BMP180.h"


N5110::N5110(PinName pwrPin, PinName scePin, PinName rstPin, PinName dcPin, PinName mosiPin, PinName sclkPin, PinName ledPin)
{

    spi = new SPI(mosiPin,NC,sclkPin); // create new SPI instance and initialise
    initSPI();

    // set up pins as required
    led = new PwmOut(ledPin);
    pwr = new DigitalOut(pwrPin);
    sce = new DigitalOut(scePin);
    rst = new DigitalOut(rstPin);
    dc = new DigitalOut(dcPin);

}

// initialise function - powers up and sends the initialisation commands
void N5110::init()
{
    turnOn();     // power up
    wait_ms(10);  // small delay seems to prevent spurious pixels during mbed reset
    reset();      // reset LCD - must be done within 100 ms

    // function set - extended
    sendCommand(0x20 | CMD_FS_ACTIVE_MODE | CMD_FS_HORIZONTAL_MODE | CMD_FS_EXTENDED_MODE);
    // Don't completely understand these parameters - they seem to work as they are
    // Consult the datasheet if you need to change them
    sendCommand(CMD_VOP_7V38);    // operating voltage - these values are from Chris Yan's Library
    sendCommand(CMD_TC_TEMP_2);   // temperature control
    sendCommand(CMD_BI_MUX_48);   // bias

    // function set - basic
    sendCommand(0x20 | CMD_FS_ACTIVE_MODE | CMD_FS_HORIZONTAL_MODE | CMD_FS_BASIC_MODE);
    normalMode();  // normal video mode by default
    sendCommand(CMD_DC_NORMAL_MODE);  // black on white

    // RAM is undefined at power-up so clear
    clearRAM();

}

// sets normal video mode (black on white)
void N5110::normalMode()
{
    sendCommand(CMD_DC_NORMAL_MODE);

}

// sets normal video mode (white on black)
void N5110::inverseMode()
{
    sendCommand(CMD_DC_INVERT_VIDEO);
}

// function to power up the LCD and backlight
void N5110::turnOn()
{
    // set brightness of LED - 0.0 to 1.0 - default is 50%
    setBrightness(0.5);
    pwr->write(1);  // apply power
}

// function to power down LCD
void N5110::turnOff()
{
    setBrightness(0.0);  // turn backlight off
    clearRAM();   // clear RAM to ensure specified current consumption
    // send command to ensure we are in basic mode
    sendCommand(0x20 | CMD_FS_ACTIVE_MODE | CMD_FS_HORIZONTAL_MODE | CMD_FS_BASIC_MODE);
    // clear the display
    sendCommand(CMD_DC_CLEAR_DISPLAY);
    // enter the extended mode and power down
    sendCommand(0x20 | CMD_FS_POWER_DOWN_MODE | CMD_FS_HORIZONTAL_MODE | CMD_FS_EXTENDED_MODE);
    // small delay and then turn off the power pin
    wait_ms(10);
    pwr->write(0);

}

// function to change LED backlight brightness
void N5110::setBrightness(float brightness)
{
    // check whether brightness is within range
    if (brightness < 0.0)
        brightness = 0.0;
    if (brightness > 1.0)
        brightness = 1.0;
    // set PWM duty cycle
    led->write(brightness);
}


// pulse the active low reset line
void N5110::reset()
{
    rst->write(0);  // reset the LCD
    rst->write(1);
}

// function to initialise SPI peripheral
void N5110::initSPI()
{
    spi->format(8,1);    // 8 bits, Mode 1 - polarity 0, phase 1 - base value of clock is 0, data captured on falling edge/propagated on rising edge
    spi->frequency(4000000);  // maximum of screen is 4 MHz
}

// send a command to the display
void N5110::sendCommand(unsigned char command)
{
    dc->write(0);  // set DC low for command
    sce->write(0); // set CE low to begin frame
    spi->write(command);  // send command
    dc->write(1);  // turn back to data by default
    sce->write(1); // set CE high to end frame (expected for transmission of single byte)

}

// send data to the display at the current XY address
// dc is set to 1 (i.e. data) after sending a command and so should
// be the default mode.
void N5110::sendData(unsigned char data)
{
    sce->write(0);   // set CE low to begin frame
    spi->write(data);
    sce->write(1);  // set CE high to end frame (expected for transmission of single byte)
}

// this function writes 0 to the 504 bytes to clear the RAM
void N5110::clearRAM()
{
    int i;
    sce->write(0);  //set CE low to begin frame
    for(i = 0; i < WIDTH * HEIGHT; i++) { // 48 x 84 bits = 504 bytes
        spi->write(0x00);  // send 0's
    }
    sce->write(1); // set CE high to end frame

}

// function to set the XY address in RAM for subsequenct data write
void N5110::setXYAddress(int x, int y)
{
    if (x>=0 && x<WIDTH && y>=0 && y<HEIGHT) {  // check within range
        sendCommand(0x80 | x);  // send addresses to display with relevant mask
        sendCommand(0x40 | y);
    }
}

// These functions are used to set, clear and get the value of pixels in the display
// Pixels are addressed in the range of 0 to 47 (y) and 0 to 83 (x).  The refresh()
// function must be called after set and clear in order to update the display
void N5110::setPixel(int x, int y)
{
    if (x>=0 && x<WIDTH && y>=0 && y<HEIGHT) {  // check within range
        // calculate bank and shift 1 to required position in the data byte
        buffer[x][y/8] |= (1 << y%8);
    }
}

void N5110::clearPixel(int x, int y)
{
    if (x>=0 && x<WIDTH && y>=0 && y<HEIGHT) {  // check within range
        // calculate bank and shift 1 to required position (using bit clear)
        buffer[x][y/8] &= ~(1 << y%8);
    }
}

int N5110::getPixel(int x, int y)
{
    if (x>=0 && x<WIDTH && y>=0 && y<HEIGHT) {  // check within range
        // return relevant bank and mask required bit
        return (int) buffer[x][y/8] & (1 << y%8);
        // note this does not necessarily return 1 - a non-zero number represents a pixel
    } else {
        return 0;
    }
}

// function to refresh the display
void N5110::refresh()
{
    int i,j;

    setXYAddress(0,0);  // important to set address back to 0,0 before refreshing display
    // address auto increments after printing string, so buffer[0][0] will not coincide
    // with top-left pixel after priting string

    sce->write(0);  //set CE low to begin frame

    for(j = 0; j < BANKS; j++) {  // be careful to use correct order (j,i) for horizontal addressing
        for(i = 0; i < WIDTH; i++) {
            spi->write(buffer[i][j]);  // send buffer
        }
    }
    sce->write(1); // set CE high to end frame

}

// fills the buffer with random bytes.  Can be used to test the display.
// The rand() function isn't seeded so it probably creates the same pattern everytime
void N5110::randomiseBuffer()
{
    int i,j;
    for(j = 0; j < BANKS; j++) {  // be careful to use correct order (j,i) for horizontal addressing
        for(i = 0; i < WIDTH; i++) {
            buffer[i][j] = rand()%256;  // generate random byte
        }
    }

}

// function to print 5x7 font
void N5110::printChar(char c,int x,int y)
{
    if (y>=0 && y<BANKS) {  // check if printing in range of y banks

        for (int i = 0; i < 5 ; i++ ) {
            int pixel_x = x+i;
            if (pixel_x > WIDTH-1)  // ensure pixel isn't outside the buffer size (0 - 83)
                break;
            buffer[pixel_x][y] = font5x7[(c - 32)*5 + i];
            // array is offset by 32 relative to ASCII, each character is 5 pixels wide
        }

        refresh();  // this sends the buffer to the display and sets address (cursor) back to 0,0
    }
}

// function to print string at specified position
void N5110::printString(const char * str,int x,int y)
{
    if (y>=0 && y<BANKS) {  // check if printing in range of y banks

        int n = 0 ; // counter for number of characters in string
        // loop through string and print character
        while(*str) {

            // writes the character bitmap data to the buffer, so that
            // text and pixels can be displayed at the same time
            for (int i = 0; i < 5 ; i++ ) {
                int pixel_x = x+i+n*6;
                if (pixel_x > WIDTH-1) // ensure pixel isn't outside the buffer size (0 - 83)
                    break;
                buffer[pixel_x][y] = font5x7[(*str - 32)*5 + i];
            }

            str++;  // go to next character in string

            n++;    // increment index

        }

        refresh();  // this sends the buffer to the display and sets address (cursor) back to 0,0
    }
}

// function to clear the screen
void N5110::clear()
{
    clearBuffer();  // clear the buffer then call the refresh function
    refresh();
}

// function to clear the buffer
void N5110::clearBuffer()
{
    int i,j;
    for (i=0; i<WIDTH; i++) {  // loop through the banks and set the buffer to 0
        for (j=0; j<BANKS; j++) {
            buffer[i][j]=0;
        }
    }
}

// function to plot array on display
void N5110::plotArray(float array[])
{

    int i;

    for (i=0; i<WIDTH; i++) {  // loop through array
        // elements are normalised from 0.0 to 1.0, so multiply
        // by 47 to convert to pixel range, and subtract from 47
        // since top-left is 0,0 in the display geometry
        setPixel(i,47 - int(array[i]*47.0));
    }

    refresh();

}

// function to draw circle
void N5110:: drawCircle(int x0,int y0,int radius,int fill)
{
    // from http://en.wikipedia.org/wiki/Midpoint_circle_algorithm
    int x = radius;
    int y = 0;
    int radiusError = 1-x;

    while(x >= y) {

        // if transparent, just draw outline
        if (fill == 0) {
            setPixel( x + x0,  y + y0);
            setPixel(-x + x0,  y + y0);
            setPixel( y + x0,  x + y0);
            setPixel(-y + x0,  x + y0);
            setPixel(-y + x0, -x + y0);
            setPixel( y + x0, -x + y0);
            setPixel( x + x0, -y + y0);
            setPixel(-x + x0, -y + y0);
        } else {  // drawing filled circle, so draw lines between points at same y value

            int type = (fill==1) ? 1:0;  // black or white fill

            drawLine(x+x0,y+y0,-x+x0,y+y0,type);
            drawLine(y+x0,x+y0,-y+x0,x+y0,type);
            drawLine(y+x0,-x+y0,-y+x0,-x+y0,type);
            drawLine(x+x0,-y+y0,-x+x0,-y+y0,type);
        }


        y++;
        if (radiusError<0) {
            radiusError += 2 * y + 1;
        } else {
            x--;
            radiusError += 2 * (y - x) + 1;
        }
    }

}

void N5110::drawLine(int x0,int y0,int x1,int y1,int type)
{
    int y_range = y1-y0;  // calc range of y and x
    int x_range = x1-x0;
    int start,stop,step;

    // if dotted line, set step to 2, else step is 1
    step = (type==2) ? 2:1;

    // make sure we loop over the largest range to get the most pixels on the display
    // for instance, if drawing a vertical line (x_range = 0), we need to loop down the y pixels
    // or else we'll only end up with 1 pixel in the x column
    if ( abs(x_range) > abs(y_range) ) {

        // ensure we loop from smallest to largest or else for-loop won't run as expected
        start = x1>x0 ? x0:x1;
        stop =  x1>x0 ? x1:x0;

        // loop between x pixels
        for (int x = start; x<= stop ; x+=step) {
            // do linear interpolation
            int y = y0 + (y1-y0)*(x-x0)/(x1-x0);

            if (type == 0)   // if 'white' line, turn off pixel
                clearPixel(x,y);
            else
                setPixel(x,y);  // else if 'black' or 'dotted' turn on pixel
        }
    } else {

        // ensure we loop from smallest to largest or else for-loop won't run as expected
        start = y1>y0 ? y0:y1;
        stop =  y1>y0 ? y1:y0;

        for (int y = start; y<= stop ; y+=step) {
            // do linear interpolation
            int x = x0 + (x1-x0)*(y-y0)/(y1-y0);

            if (type == 0)   // if 'white' line, turn off pixel
                clearPixel(x,y);
            else
                setPixel(x,y);  // else if 'black' or 'dotted' turn on pixel

        }
    }

}

void N5110::drawRect(int x0,int y0,int width,int height,int fill)
{

    if (fill == 0) { // transparent, just outline
        drawLine(x0,y0,x0+width,y0,1);  // top
        drawLine(x0,y0+height,x0+width,y0+height,1);  // bottom
        drawLine(x0,y0,x0,y0+height,1);  // left
        drawLine(x0+width,y0,x0+width,y0+height,1);  // right
    } else { // filled rectangle
        int type = (fill==1) ? 1:0;  // black or white fill
        for (int y = y0; y<= y0+height; y++) {  // loop through rows of rectangle
            drawLine(x0,y,x0+width,y,type);  // draw line across screen
        }
    }

}



BMP180::BMP180(PinName sdaPin, PinName sclPin)
{
    i2c = new I2C(sdaPin,sclPin); // create new I2C instance and initialise
    i2c->frequency(400000);       // I2C Fast Mode - 400kHz
    leds = new BusOut(LED4,LED3,LED2,LED1);
}

Measurement BMP180::readValues()
{
    // algorithm for taking measurement is taken from datasheet
    int32_t UT = readUncompensatedTemperatureValue();
    int32_t UP = readUncompensatedPressureValue();
    // once you have the uncompensated T and P, you can calculate the true T and P
    // using the equations from the datasheet
    int32_t T = calcTrueTemperature(UT);
    int32_t P = calcTruePressure(UP);

    Measurement measurement;
    measurement.temperature = T*0.1;  // scaled by 0.1 C
    measurement.pressure = P*0.01;    // Put pressure in mb

    return measurement;
}

int32_t BMP180::readUncompensatedTemperatureValue()
{
    // from algorithm in datasheet - p15
    sendByteToRegister(0x2E,0xF4);
    wait_ms(5);  // 4.5 ms delay for OSS = 1
    char MSB = readByteFromRegister(0xF6);
    char LSB = readByteFromRegister(0xF7);
    // combine in 16-bit value
    int UT = (MSB << 8) | LSB;
#ifdef DEBUG
    UT = 27898;  // test data from datasheet
    printf("****DEBUG MODE****\nUT = %d\n",UT);
#endif
    return UT;
}

int32_t BMP180::readUncompensatedPressureValue()
{
    // from datasheet
    char byte = 0x34 + (oss << 6);
    sendByteToRegister(byte,0xF4);
    wait_ms(8);  // 7.5 ms delay for OSS = 1

    char MSB = readByteFromRegister(0xF6);
    char LSB = readByteFromRegister(0xF7);
    char XLSB = readByteFromRegister(0xF7);
    int UP = (MSB << 16 | LSB << 8 | XLSB) >> (8 - oss);

#ifdef DEBUG
    UP = 23843;   // test data from datasheet
    printf("UP = %d\n",UP);
#endif
    return UP;
}

int32_t BMP180::calcTrueTemperature(int32_t UT)
{
    // equations from data sheet
    X1 = ((UT - calibration.AC6)*calibration.AC5) >> 15;
    X2 = (calibration.MC << 11) / (X1 + calibration.MD);
    B5 = X1 + X2;
    int32_t T = (B5 + 8) >> 4;
#ifdef DEBUG
    printf("****\nX1=%d\nX2=%d\nB5=%d\nT=%d\n",X1,X2,B5,T);
#endif
    return T;
}

int32_t BMP180::calcTruePressure(int32_t UP)
{
    // equations from data sheet
    B6 = B5 - 4000;
    X1 = (calibration.B2 * ((B6*B6) >> 12))>>11;
    X2 = (calibration.AC2*B6)>>11;
    X3 = X1 + X2;
    B3 = (((calibration.AC1*4 + X3) << oss)+2)/4;
#ifdef DEBUG
    printf("*****\nB6=%d\nX1=%d\nX2=%d\nX3=%d\nB3=%d\n",B6,X1,X2,X3,B3);
#endif
    X1 = (calibration.AC3*B6)>>13;
    X2 = (calibration.B1*((B6*B6)>>12))>>16;
    X3 = ((X1+X2)+2)/4;
    B4 = (calibration.AC4*(uint32_t)(X3+32768))>>15;
#ifdef DEBUG
    printf("X1=%d\nX2=%d\nX3=%d\nB4=%u\n",X1,X2,X3,B4);
#endif
    B7 = ((uint32_t)UP - B3)*(50000>>oss);
#ifdef DEBUG
    printf("B7=%u\n",B7);
#endif
    int32_t P;
    if (B7 < 0x80000000)
        P = (B7*2)/B4;
    else
        P = (B7/B4)*2;
#ifdef DEBUG
    printf("P=%d\n",P);
#endif
    X1 = (P>>8)*(P>>8);
#ifdef DEBUG
    printf("X1=%d\n",X1);
#endif
    X1 = (X1*3038)>>16;
#ifdef DEBUG
    printf("X1=%d\n",X1);
#endif
    X2 = (-7357*P)>>16;
#ifdef DEBUG
    printf("X2=%d\n",X2);
#endif
    P = P + (X1+X2+3791)/16;
#ifdef DEBUG
    printf("P=%d\n",P);
#endif

    return P;

}

// configure the barometer
void BMP180::init()
{
    i2c->frequency(400000); // set Fast Mode I2C frequency

    char data = readByteFromRegister(ID_REG);  // Section 4 - datasheet
    if (data != 0x55) { // if correct ID not found, hang and flash error message
        error();
    }

    readCalibrationData();

    oss = 1;  // standard power oversampling setting

#ifdef DEBUG
    oss = 0;  // used when testing data sheet example
#endif


}

// Reads factory calibrated data
void BMP180::readCalibrationData()
{

    char eeprom[22];

    readBytesFromRegister(EEPROM_REG_ADD,22,eeprom);
    // store calibration data in structure
    calibration.AC1 = (int16_t) (eeprom[0] << 8) | eeprom[1];
    calibration.AC2 = (int16_t) (eeprom[2] << 8) | eeprom[3];
    calibration.AC3 = (int16_t) (eeprom[4] << 8) | eeprom[5];
    calibration.AC4 = (uint16_t) (eeprom[6] << 8) | eeprom[7];
    calibration.AC5 = (uint16_t) (eeprom[8] << 8) | eeprom[9];
    calibration.AC6 = (uint16_t) (eeprom[10] << 8) | eeprom[11];
    calibration.B1 = (int16_t) (eeprom[12] << 8) | eeprom[13];
    calibration.B2 = (int16_t) (eeprom[14] << 8) | eeprom[15];
    calibration.MB = (int16_t) (eeprom[16] << 8) | eeprom[17];
    calibration.MC = (int16_t) (eeprom[18] << 8) | eeprom[19];
    calibration.MD = (int16_t) (eeprom[20] << 8) | eeprom[21];

    // test data from data sheet
#ifdef DEBUG
    calibration.AC1 = 408;
    calibration.AC2 = -72;
    calibration.AC3 = -14383;
    calibration.AC4 = 32741;
    calibration.AC5 = 32757;
    calibration.AC6 = 23153;
    calibration.B1 = 6190;
    calibration.B2 = 4;
    calibration.MB = -32768;
    calibration.MC = -8711;
    calibration.MD = 2868;
    printf("****EXAMPLE CALIBRATION DATA****\n");
    printf("AC1=%d\nAC2=%d\nAC3=%d\nAC4=%u\nAC5=%u\nAC6=%u\nB1=%d\nB2=%d\nMB=%d\nMC=%d\nMD=%d\n",
           calibration.AC1,calibration.AC2,calibration.AC3,calibration.AC4,calibration.AC5,calibration.AC6,
           calibration.B1,calibration.B2,calibration.MB,calibration.MC,calibration.MD);
#endif
}


// reads a byte from a specific register
char BMP180::readByteFromRegister(char reg)
{
    int nack = i2c->write(BMP180_W_ADDRESS,&reg,1,true);  // send the register address to the slave
    if (nack)
        error();  // if we don't receive acknowledgement, flash error message

    char rx;
    nack = i2c->read(BMP180_W_ADDRESS,&rx,1);  // read a byte from the register and store in buffer
    if (nack)
        error();  // if we don't receive acknowledgement, flash error message

    return rx;
}

// reads a series of bytes, starting from a specific register
void BMP180::readBytesFromRegister(char reg,int numberOfBytes,char bytes[])
{
    int nack = i2c->write(BMP180_W_ADDRESS,&reg,1,true);  // send the slave write address and the configuration register address

    if (nack)
        error();  // if we don't receive acknowledgement, flash error message

    nack = i2c->read(BMP180_W_ADDRESS,bytes,numberOfBytes);  // read bytes
    if (nack)
        error();  // if we don't receive acknowledgement, flash error message

}

// sends a byte to a specific register
void BMP180::sendByteToRegister(char byte,char reg)
{
    char data[2];
    data[0] = reg;
    data[1] = byte;
    // send the register address, followed by the data
    int nack = i2c->write(BMP180_W_ADDRESS,data,2);
    if (nack)
        error();  // if we don't receive acknowledgement, flash error message

}

void BMP180::error()
{
    while(1) {
        leds->write(15);
        wait(0.1);
        leds->write(0);
        wait(0.1);
    }
}




N5110 lcd(p7,p8,p9,p10,p11,p13,p26);//The ports  being connected of the mbed
BusOut leds(LED4,LED3,LED2,LED1);//The ports  being connected of the LCD
BMP180 bmp180(p28,p27);//The ports being connected to the sensor
Serial serial(USBTX,USBRX);//Timer set-up tool


//Define the variable
void serialISR();//ISR that is called when serial data is received
void setTime();// function to set the UNIX time
int setTimerFlag = 0;// flag for ISR
char rxString[16];//Create a 16 chars row to display the data

int main()
{
    lcd.init();
    bmp180.init();//Display the word before the sensor data
    lcd.printString("Weather",0,0);//At the location (0,0),display word "Weather"
    lcd.printString("Station",1,3);//At the location (1,3),display word "Station"
    wait(2.0);//The word above stay for 2s
    lcd.clear();//Clean the display for the continued work
  
    Measurement measurement;

    serial.attach(&serialISR);//attach serial ISR
    char t[30];//Create a 30 chars row to display the time
    while(1) {

        time_t seconds = time(NULL);//get current time
        // format time into a string (time and date)
        strftime(t, 30 , "%X %D",localtime(&seconds));//
        // print over serial
        serial.printf("Time = %s\n" ,t);//Display the timer
        lcd.printString(t,0,5);//The location of the timer

        if(setTimerFlag) {// if updated time has been sent
            setTimerFlag = 0;//clear flag
            setTime();// update time

        }

        measurement = bmp180.readValues();//
        char T[14];//Create a 14 chars row to display the temperature
        int length =sprintf(T,"T = %.2f C",measurement.temperature);//Set up "T = sensor data" as the thing will be shown
        if (length <= 14)//Judge the length of chars
        lcd.printString(T,0,1);//The location of the T will be shown
        char P[14];//Create a 14 chars row to display the pressure
        length = sprintf(P,"P = %.2f mb",measurement.pressure);//Set up "P = sensor data" as the thing will be shown
        lcd.printString(P,0,3); //The location of the P will be shown
        wait(1);//Repeat the circulate each 1s
        lcd.clear();  //Clear the data for next processing
    }     
}


void setTime()//// print time for debugging
{
     
    serial.printf("set_time - %s",rxString);
    //// atoi() converts a string to an integer
    int time = atoi(rxString);
    //update the time
    set_time(time);
}

void serialISR()// when a serial interrupt occurs, read rx string into buffer
{   
    
    serial.gets(rxString,16);
    //// set flag
    setTimerFlag = 1;
}
