/// high-bandwidth 3-phase motor control, for robots
/// Written by benkatz, with much inspiration from Bayley Wang, Nick Kirkby, Shane Colton, David Otten, and others
/// Hardware documentation can be found at build-its.blogspot.com
/// Written for the STM32F446, but can be implemented on other STM32 MCU's with some further register-diddling
/// Version for the TI DRV8323 Everything Chip

#define REST_MODE 0
#define CALIBRATION_MODE 1
#define MOTOR_MODE 2
#define SETUP_MODE 4
#define ENCODER_MODE 5

#define VERSION_NUM "0.3"


float __float_reg[64];                                                          // Floats stored in flash
int __int_reg[256];                                                             // Ints stored in flash.  Includes position sensor calibration lookup table

#include "mbed.h"
#include "PositionSensor.h"
#include "structs.h"
#include "foc.h"
#include "calibration.h"
#include "hw_setup.h"
#include "math_ops.h" 
#include "current_controller_config.h"
#include "hw_config.h"
#include "motor_config.h"
#include "stm32f4xx_flash.h"
#include "FlashWriter.h"
#include "user_config.h"
#include "PreferenceWriter.h"
#include "CAN_com.h"
#include "DRV.h"
 
PreferenceWriter prefs(6);

GPIOStruct gpio;
ControllerStruct controller;
ObserverStruct observer;
COMStruct com;
Serial pc(PA_2, PA_3);


CAN          can(PB_8, PB_9, 1000000);      // CAN Rx pin name, CAN Tx pin name
CANMessage   rxMsg;
CANMessage   txMsg;


SPI drv_spi(PA_7, PA_6, PA_5);
DigitalOut drv_cs(PA_4);
//DigitalOut drv_en_gate(PA_11);
DRV832x drv(&drv_spi, &drv_cs);

PositionSensorAM5147 spi(16384, 0.0, NPP);  

volatile int counts = 0;
volatile int state = REST_MODE;
volatile int state_change;


volatile int cal_counts = 0;

void onMsgReceived() {
    //msgAvailable = true;
    //printf("%d\n\r", rxMsg.id);
    can.read(rxMsg);  
    if((rxMsg.id == CAN_ID)){
        controller.timeout = 0;
        if(((rxMsg.data[0]==0xFF) & (rxMsg.data[1]==0xFF) & (rxMsg.data[2]==0xFF) & (rxMsg.data[3]==0xFF) & (rxMsg.data[4]==0xFF) & (rxMsg.data[5]==0xFF) & (rxMsg.data[6]==0xFF) & (rxMsg.data[7]==0xFC))){
            state = MOTOR_MODE;
            state_change = 1;
            }
        else if(((rxMsg.data[0]==0xFF) & (rxMsg.data[1]==0xFF) & (rxMsg.data[2]==0xFF) & (rxMsg.data[3]==0xFF) * (rxMsg.data[4]==0xFF) & (rxMsg.data[5]==0xFF) & (rxMsg.data[6]==0xFF) & (rxMsg.data[7]==0xFD))){
            state = REST_MODE;
            state_change = 1;
            gpio.led->write(0);; 
            }
        else if(((rxMsg.data[0]==0xFF) & (rxMsg.data[1]==0xFF) & (rxMsg.data[2]==0xFF) & (rxMsg.data[3]==0xFF) * (rxMsg.data[4]==0xFF) & (rxMsg.data[5]==0xFF) & (rxMsg.data[6]==0xFF) & (rxMsg.data[7]==0xFE))){
            spi.ZeroPosition();
            }            
        else if(((rxMsg.data[0]==0xFF) & (rxMsg.data[1]==0xFF) & (rxMsg.data[2]==0xFF) & (rxMsg.data[3]==0xFF) * (rxMsg.data[4]==0xFF) & (rxMsg.data[5]==0xFF) & (rxMsg.data[6]==0xFF) & (rxMsg.data[7]==0xFF))){
            //Null instruction, which can be used to return data
            }
        else if(state == MOTOR_MODE){
            unpack_cmd(rxMsg, &controller);
            }
        //pack_reply(&txMsg, controller.theta_mech, controller.dtheta_mech, controller.i_q_filt*KT_OUT);
        //pack_reply(&txMsg, controller.theta_mech, controller.dtheta_mech*GR, controller.i_q_filt*KT_OUT);    // modify by 20210617
        //pack_reply(&txMsg, controller.theta_mech, controller.dtheta_mech*GR, controller.i_q_filt*KT);   // 正式使用 plan A&C, modify by 20210630        
        pack_reply(&txMsg, controller.theta_mech, controller.dtheta_mech, controller.i_q_filt*KT);      //  test v?, modify by 20210706     
        can.write(txMsg);
        }
    
}

void enter_menu_state(void){
    drv.disable_gd();
    //gpio.enable->write(0);
    printf("\n\r\n\r\n\r");
    printf(" Commands:\n\r");
    wait_us(10);
    printf(" m - Motor Mode\n\r");
    wait_us(10);
    printf(" c - Calibrate Encoder\n\r");
    wait_us(10);
    printf(" s - Setup\n\r");
    wait_us(10);
    printf(" e - Display Encoder\n\r");
    wait_us(10);
    printf(" z - Set Zero Position\n\r");
    wait_us(10);
    printf(" esc - Exit to Menu\n\r");
    wait_us(10);
    state_change = 0;
    gpio.led->write(0);
    }

void enter_setup_state(void){
    printf("\n\r\n\r Configuration Options \n\r\n\n");
    wait_us(10);
    printf(" %-4s %-31s %-5s %-6s %-2s\n\r\n\r", "prefix", "parameter", "min", "max", "current value");
    wait_us(10);
    printf(" %-4s %-31s %-5s %-6s %.1f\n\r", "b", "Current Bandwidth (Hz)", "100", "2000", I_BW);
    wait_us(10);
    printf(" %-4s %-31s %-5s %-6s %-5i\n\r", "i", "CAN ID", "0", "127", CAN_ID);
    wait_us(10);
    printf(" %-4s %-31s %-5s %-6s %-5i\n\r", "m", "CAN Master ID", "0", "127", CAN_MASTER);
    wait_us(10);
    printf(" %-4s %-31s %-5s %-6s %.1f\n\r", "l", "Current Limit (A)", "0.0", "40.0", I_MAX);
    wait_us(10);
    printf(" %-4s %-31s %-5s %-6s %.1f\n\r", "f", "FW Current Limit (A)", "0.0", "33.0", I_FW_MAX);
    wait_us(10);
    printf(" %-4s %-31s %-5s %-6s %d\n\r", "t", "CAN Timeout (cycles)(0 = none)", "0", "100000", CAN_TIMEOUT);
    wait_us(10);
    printf("\n\r To change a value, type 'prefix''value''ENTER'\n\r i.e. 'b1000''ENTER'\n\r\n\r");
    wait_us(10);
    state_change = 0;
    }
    
void enter_torque_mode(void){
    drv.enable_gd();
    //gpio.enable->write(1);
    controller.ovp_flag = 0;
    reset_foc(&controller);                                                     // Tesets integrators, and other control loop parameters
    wait(.001);
    controller.i_d_ref = 0;
    controller.i_q_ref = 0;                                                     // Current Setpoints
    gpio.led->write(1);                                                     // Turn on status LED
    state_change = 0;
    printf("\n\r Entering Motor Mode \n\r");
    }
    
void calibrate(void){
    drv.enable_gd();
    //gpio.enable->write(1);
    gpio.led->write(1);                                                    // Turn on status LED
    order_phases(&spi, &gpio, &controller, &prefs);                             // Check phase ordering
    calibrate(&spi, &gpio, &controller, &prefs);                                // Perform calibration procedure
    gpio.led->write(0);;                                                     // Turn off status LED
    wait(.2);
    printf("\n\r Calibration complete.  Press 'esc' to return to menu\n\r");
    drv.disable_gd();
    //gpio.enable->write(0);
     state_change = 0;
    }
    
void print_encoder(void){
    printf(" Mechanical Angle:  %f    Electrical Angle:  %f    Raw:  %d\n\r", spi.GetMechPosition(), spi.GetElecPosition(), spi.GetRawPosition());
    //printf("%d\n\r", spi.GetRawPosition());
    wait(.001);
    }

/// Current Sampling Interrupt ///
/// This runs at 40 kHz, regardless of of the mode the controller is in ///
extern "C" void TIM1_UP_TIM10_IRQHandler(void) {
  if (TIM1->SR & TIM_SR_UIF ) {
        ///Sample current always ///
        ADC1->CR2  |= 0x40000000;                                               // Begin sample and conversion
        //volatile int delay;   
        //for (delay = 0; delay < 55; delay++);

        spi.Sample(DT);                                                           // sample position sensor
        controller.adc2_raw = ADC2->DR;                                         // Read ADC Data Registers
        controller.adc1_raw = ADC1->DR;
        controller.adc3_raw = ADC3->DR;
        controller.theta_elec = spi.GetElecPosition();
        //controller.theta_mech = (1.0f/GR)*spi.GetMechPosition();
        controller.dtheta_mech = (1.0f/GR)*spi.GetMechVelocity();   
        controller.theta_mech = (1.0f)*spi.GetMechPosition();    // modified on 20210617
        //controller.dtheta_mech = (1.0f)*spi.GetMechVelocity();   // modified on 20210618
        
        controller.dtheta_elec = spi.GetElecVelocity();
        controller.v_bus = 0.95f*controller.v_bus + 0.05f*((float)controller.adc3_raw)*V_SCALE; //filter the dc link voltage measurement
        ///
        
        /// Check state machine state, and run the appropriate function ///
        switch(state){
            case REST_MODE:                                                     // Do nothing
                if(state_change){
                    enter_menu_state();
                    }
                break;
            
            case CALIBRATION_MODE:                                              // Run encoder calibration procedure
                if(state_change){
                    calibrate();
                    }
                break;
             
            case MOTOR_MODE:                                                   // Run torque control
                if(state_change){
                    
                    enter_torque_mode();
                    
                    counts = 0;  //20210222
                    
                    }
                else{
                /*
                if(controller.v_bus>28.0f){         //Turn of gate drive if bus voltage is too high, to prevent FETsplosion if the bus is cut during regen
                    gpio.
                    ->write(0);
                    controller.ovp_flag = 1;
                    state = REST_MODE;
                    state_change = 1;
                    printf("OVP Triggered!\n\r");
                    }
                    */  

                if((controller.timeout > CAN_TIMEOUT) && (CAN_TIMEOUT > 0)){
                    controller.i_d_ref = 0;
                    controller.i_q_ref = 0;
                    controller.kp = 0;
                    controller.kd = 0;
                    controller.t_ff = 0;
                    } 

                torque_control(&controller);
                commutate(&controller, &observer, &gpio, controller.theta_elec);           // Run current loop
                controller.timeout++;
                counts ++; 
            
                }     
                break;
            case SETUP_MODE:
                if(state_change){
                    enter_setup_state();
                }
                break;
            case ENCODER_MODE:
                print_encoder();
                break;
                }    
      }
  TIM1->SR = 0x0;                                                               // reset the status register
}


char cmd_val[8] = {0};
char cmd_id = 0;
char char_count = 0;

/// Manage state machine with commands from serial terminal or configurator gui ///
/// Called when data received over serial ///
void serial_interrupt(void){
    while(pc.readable()){
        char c = pc.getc();
        if(c == 27){
                state = REST_MODE;
                state_change = 1;
                char_count = 0;
                cmd_id = 0;
                gpio.led->write(0);; 
                for(int i = 0; i<8; i++){cmd_val[i] = 0;}
                }
        if(state == REST_MODE){
            switch (c){
                case 'c':
                    state = CALIBRATION_MODE;
                    state_change = 1;
                    break;
                case 'm':
                    state = MOTOR_MODE;
                    state_change = 1;                    
                    break;
                case 'e':
                    state = ENCODER_MODE;
                    state_change = 1;
                    break;
                case 's':
                    state = SETUP_MODE;
                    state_change = 1;
                    break;
                case 'z':
                    spi.SetMechOffset(0);
                    spi.Sample(DT);
                    wait_us(20);
                    M_OFFSET = spi.GetMechPosition();
                    if (!prefs.ready()) prefs.open();
                        prefs.flush();                                                  // Write new prefs to flash
                        prefs.close();    
                        prefs.load(); 
                    spi.SetMechOffset(M_OFFSET);
                    printf("\n\r  Saved new zero position:  %.4f\n\r\n\r", M_OFFSET);
                    
                    break;
                }
                
                }
        else if(state == SETUP_MODE){
            if(c == 13){
                switch (cmd_id){
                    case 'b':
                        I_BW = fmaxf(fminf(atof(cmd_val), 2000.0f), 100.0f);
                        break;
                    case 'i':
                        CAN_ID = atoi(cmd_val);
                        break;
                    case 'm':
                        CAN_MASTER = atoi(cmd_val);
                        break;
                    case 'l':
                        I_MAX = fmaxf(fminf(atof(cmd_val), 40.0f), 0.0f);
                        break;
                    case 'f':
                        I_FW_MAX = fmaxf(fminf(atof(cmd_val), 33.0f), 0.0f);
                        break;
                    case 't':
                        CAN_TIMEOUT = atoi(cmd_val);
                        break;
                    default:
                        printf("\n\r '%c' Not a valid command prefix\n\r\n\r", cmd_id);
                        break;
                    }
                    
                if (!prefs.ready()) prefs.open();
                prefs.flush();                                                  // Write new prefs to flash
                prefs.close();    
                prefs.load();                                              
                state_change = 1;
                char_count = 0;
                cmd_id = 0;
                for(int i = 0; i<8; i++){cmd_val[i] = 0;}
                }
            else{
                if(char_count == 0){cmd_id = c;}
                else{
                    cmd_val[char_count-1] = c;
                    
                }
                pc.putc(c);
                char_count++;
                }
            }
        else if (state == ENCODER_MODE){
            switch (c){
                case 27:
                    state = REST_MODE;
                    state_change = 1;
                    break;
                    }
            }
        else if (state == MOTOR_MODE){
            switch (c){
                case 'd':
                    controller.i_q_ref = 0;
                    controller.i_d_ref = 0;
                }
            }
            
        }
    }
    
//PT1000用二分法查表计算温度
//测温范围：-50~150℃
float RTD_TAB_PT1000[201] = {

803.08,807.05,811.02,814.98,818.95,822.92,826.88,830.84,834.80,838.76,              //-50~-41
842.72,846.68,850.63,854.58,858.54,862.49,866.44,870.39,874.33,878.28,              //-40~-31
882.23,886.17,890.11,894.05,897.99,901.93,905.87,909.80,913.74,917.67,              //-30~-21
921.60,925.54,929.47,933.39,937.32,941.25,945.17,949.10,953.02,956.94,              //-20~-11
960.86,964.78,968.70,972.62,976.53,980.45,984.36,988.27,992.18,996.09,              //-10~-1
1000.00,1003.91,1007.81,1011.72,1015.62,1019.53,1023.43,1027.33,1031.23,1035.13,    //0~9
1039.02,1042.92,1046.81,1050.71,1054.60,1058.49,1062.38,1066.27,1070.16,1074.04,    //10~19
1077.93,1081.81,1085.70,1089.58,1093.46,1097.34,1101.22,1105.10,1108.97,1112.85,    //20~29
1116.72,1120.59,1124.46,1128.34,1132.20,1136.07,1139.94,1143.81,1147.67,1151.53,    //30~39
1155.40,1159.26,1163.12,1166.98,1170.83,1174.69,1178.55,1182.40,1186.25,1190.11,    //40~49
1193.96,1197.81,1201.65,1205.50,1209.35,1213.19,1217.04,1220.88,1224.72,1228.56,    //50~59
1232.40,1236.24,1240.08,1243.91,1247.75,1251.58,1255.41,1259.24,1263.07,1266.90,    //60~69
1270.73,1274.56,1278.38,1282.21,1286.03,1289.85,1293.67,1297.49,1301.31,1305.13,    //70~79
1308.94,1312.76,1316.57,1320.39,1324.20,1328.01,1331.82,1335.62,1339.43,1343.24,    //80~89
1347.04,1350.85,1354.65,1358.45,1362.25,1366.05,1369.85,1373.64,1377.44,1381.23,    //90~99
1385.03,1388.82,1392.61,1396.40,1400.19,1403.97,1407.76,1411.54,1415.33,1419.11,    //100~109
1422.89,1426.67,1430.45,1434.23,1438.01,1441.78,1445.56,1449.33,1453.10,1456.87,    //110~119
1460.64,1464.41,1468.18,1471.95,1475.71,1479.48,1483.24,1487.00,1490.76,1494.52,    //120~129
1498.28,1502.04,1505.79,1509.55,1513.30,1517.06,1520.81,1524.56,1528.31,1532.05,    //130~139
1535.80,1539.55,1543.29,1547.03,1550.78,1554.52,1558.26,1562.00,1565.73,1569.47,    //140~149
1573.21                                                                             //150
};

/********************************************
功能：采用2分查找法,根据电阻值查表计算温度值
输入：fR,PT1000的电阻值
输出：fTemp,测量的温度值
********************************************/
float CalTemp(float fR)
{
    float fTemp;
    float fLowR;
    float fHighR;
    int iTemp;
    int i;
    int Top,Bottom;
    
    cal_counts=0;
    
    if(fR < RTD_TAB_PT1000[0])    //电阻值小于表格最小值
    {
        return -50;
    }
    
    if(fR > RTD_TAB_PT1000[200])    //电阻值大于表格最大值
    {
        return 150;
    }
    
    Bottom = 0;
    Top = 200;
    
    for(i=100;(Top-Bottom)!=1;)    //2分法查表
    {
        if(fR < RTD_TAB_PT1000[i])
        {
            Top = i;
            i = (Top + Bottom) / 2;
        }
        else if(fR > RTD_TAB_PT1000[i])
        {
            Bottom = i;
            i = (Top + Bottom) / 2;        
        }
        else
        {
            iTemp = i - 50;
            fTemp = (float)iTemp;
            return fTemp;
        }    
        cal_counts++;
    }
    
    iTemp = i - 50;
    fLowR = RTD_TAB_PT1000[Bottom];
    fHighR = RTD_TAB_PT1000[Top];
    fTemp = (fR - fLowR)/(fHighR - fLowR) + iTemp;
    return fTemp;
}

    
    
    
    
       
int main() {
    //*************************
    //uint32_t    btr;    
    //CAN_TypeDef *can = obj->CanHandle.Instance;
    /*  
    hcan->Instance->BTR = (uint32_t)((uint32_t)hcan->Init.Mode) | \
                ((uint32_t)hcan->Init.SJW) | \
                ((uint32_t)hcan->Init.BS1) | \
                ((uint32_t)hcan->Init.BS2) | \
               ((uint32_t)hcan->Init.Prescaler - 1U);
    */
    
    //uint32_t i,num;
    float ii;
    
    
    controller.v_bus = V_BUS;
    controller.mode = 0;
    Init_All_HW(&gpio);                                                         // Setup PWM, ADC, GPIO
    wait(.1);
    
    gpio.enable->write(1);
    wait_us(100);
    drv.calibrate();
    wait_us(100);
    drv.write_DCR(0x0, 0x0, 0x0, PWM_MODE_3X, 0x0, 0x0, 0x0, 0x0, 0x1);
    wait_us(100);
    drv.write_CSACR(0x0, 0x1, 0x0, CSA_GAIN_40, 0x0, 0x0, 0x0, 0x0, SEN_LVL_1_0);
    wait_us(100);
    drv.write_OCPCR(TRETRY_4MS, DEADTIME_200NS, OCP_RETRY, OCP_DEG_8US, VDS_LVL_1_88);
    
    //drv.enable_gd();
    zero_current(&controller.adc1_offset, &controller.adc2_offset);             // Measure current sensor zero-offset
    drv.disable_gd();

    wait(.1);
    /*
    gpio.enable->write(1);
    TIM1->CCR3 = 0x708*(1.0f);                        // Write duty cycles
    TIM1->CCR2 = 0x708*(1.0f);
    TIM1->CCR1 = 0x708*(1.0f);
    gpio.enable->write(0);
    */
    reset_foc(&controller);                                                     // Reset current controller
    reset_observer(&observer);                                                 // Reset observer
    TIM1->CR1 ^= TIM_CR1_UDIS;
    //TIM1->CR1 |= TIM_CR1_UDIS; //enable interrupt
    
    wait(.1);
    NVIC_SetPriority(TIM1_UP_TIM10_IRQn, 2);                                             // commutation > communication
    
    NVIC_SetPriority(CAN1_RX0_IRQn, 3);
                                   // attach 'CAN receive-complete' interrupt handler    
    
    // If preferences haven't been user configured yet, set defaults 
    prefs.load();                                                               // Read flash
    
    can.filter(CAN_ID , 0xFFF, CANStandard, 0);
                                                                    
    txMsg.id = CAN_MASTER;
    txMsg.len = 6;
    rxMsg.len = 8;
    can.attach(&onMsgReceived);  
    
    if(isnan(E_OFFSET)){E_OFFSET = 0.0f;}
    if(isnan(M_OFFSET)){M_OFFSET = 0.0f;}
    if(isnan(I_BW) || I_BW==-1){I_BW = 1000;}
    if(isnan(I_MAX) || I_MAX ==-1){I_MAX=30;}
    if(isnan(I_FW_MAX) || I_FW_MAX ==-1){I_FW_MAX=0;}
    if(isnan(CAN_ID) || CAN_ID==-1){CAN_ID = 1;}
    if(isnan(CAN_MASTER) || CAN_MASTER==-1){CAN_MASTER = 0;}
    if(isnan(CAN_TIMEOUT) || CAN_TIMEOUT==-1){CAN_TIMEOUT = 0;}
    spi.SetElecOffset(E_OFFSET);                                                // Set position sensor offset
    spi.SetMechOffset(M_OFFSET);
    int lut[128] = {0};
    memcpy(&lut, &ENCODER_LUT, sizeof(lut));
    spi.WriteLUT(lut);                                                          // Set potision sensor nonlinearity lookup table
    init_controller_params(&controller);

    pc.baud(921600);                                                            // set serial baud rate
    pc.attach(&serial_interrupt);                                               // attach serial interrupt
    wait(.01);
    pc.printf("\n\r\n\r GTGJ MOTOR\n\r\n\r");
    wait(.01);
    printf("\n\r Debug Info:\n\r");
    printf(" Firmware Version: %s\n\r", VERSION_NUM);
    printf(" ADC1 Offset: %d    ADC2 Offset: %d\n\r", controller.adc1_offset, controller.adc2_offset);
    printf(" Position Sensor Electrical Offset:   %.4f\n\r", E_OFFSET);
    printf(" Output Zero Position:  %.4f\n\r", M_OFFSET);
    printf(" CAN ID:  %d\n\r", CAN_ID);
    

//*********
    //printf(" %d\n\r", drv.read_register(DCR));
    //wait_us(100);
    //printf(" %d\n\r", drv.read_register(CSACR));
    //wait_us(100);
    //printf(" %d\n\r", drv.read_register(OCPCR));
    //drv.disable_gd();
    
//    pc.attach(&serial_interrupt);                                               // attach serial interrupt
    
    state_change = 1;


    //**********************************************************************************************
    //for(i=800;i<1580;i++)
    for(ii=1600;ii<3150;ii++)
    {
         //printf("%d   %d   %.2f\n\r", ii/2, cal_counts, CalTemp((float)ii/2));    
         printf("%.1f   %d   %.2f\n\r", ii/2.0f, cal_counts, CalTemp(ii/2.0f));    
    }
   
    //**********************************************************************************************

//    int counter = 0;
    while(1) {
        drv.print_faults();
        wait(.1);
        //printf("%.4f\n\r", controller.v_bus);
       /*
        if(state == MOTOR_MODE)
        {
            //printf("%.3f  %.3f  %.3f\n\r", (float)observer.temperature, (float)observer.temperature2, observer.resistance);
            //printf("%.3f  %.3f  %.3f %.3f %.3f %.3f\n\r", controller.v_d, controller.v_q, controller.i_d_filt, controller.i_q_filt, controller.i_q_ref,controller.dtheta_elec);
            //printf("%.3f  %.3f  %.3f %.3f %.3f %.3f %.3f %.3f\n\r", controller.i_a, controller.i_b, controller.i_c,controller.theta_elec,controller.i_d, controller.i_q,controller.i_d_filt, controller.i_q_filt);
            printf("%.3f  %.3f  %.3f %.3f %.3f %.3f %.3f %.3f %.3f\n\r", controller.v_d, controller.v_q, controller.i_d_filt, controller.i_q_filt, controller.i_q_ref,controller.theta_elec,controller.theta_mech, controller.dtheta_mech*GR, controller.i_q_filt*KT_OUT/GR);
            //printf("%.3f\n\r", controller.dtheta_mech);            
            //printf("%d   %.4f    %.4f\n\r", spi.GetRawPosition(), spi.GetMechPositionFixed(),spi.GetMechPosition()); 
            wait(.005);
        }
        */

    }
}
