/*
 * Copyright (c) 2011 Paul van der Wielen, Pro-Serv
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to use
 * and implement the software for none commercial reason and usage only and
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 * Usage and assumptions:
 * a RFID reader Model MF7(1.05) is configured in wiegand mode and attached 
 * ports per call to class (could be to p5 & p6 via a level converter as to
 * isolate the 5V source voltage from the 3.3V needed by MCU, this can easily
 * be done with diode in series with signal data0 and data1 of RFID reader and
 * pull up on MCU side.
 *
 * Note:    using 'Wiegand' format rfid reader has one drawback as it only is capable
 *          of returning 6 digits out of the 8 digits of the full RFID code, which is
 *          the case for an 1K MiFare card.
 *          Benefit is that a lot of Wiegand compatible reader do exist from various
 *          brands at an reasonable price.
 */
 
#include "mbed.h"
#define W_PULSES 26

class Wiegand {
public:
    Wiegand(PinName pin0, PinName pin1) : _D0(pin0), _D1(pin1) { // create the InterruptIn on the pin specified to Wiegand
        _D0.mode(PullUp);
        _D1.mode(PullUp);
        _D0.rise(this, &Wiegand::data_0); // attach data_0 function of this wiegand instance
        _D1.rise(this, &Wiegand::data_1); // attach data_1 function of this wiegand instance         
    }

    /*
     * Interrupt driven signal from data0 data line
     */
    void data_0() {
        if (_pulses == W_PULSES) {
            clear();
        }
        _wiegand = ((_wiegand << 1) & 0x3ffffff) + 0x00;
        if (_pulses < W_PULSES / 2) {
            _e_par = (_e_par & 0x01);
        } else {
            _e_par = (_e_par & 0x01);
        }
        _pulses++;
    }
    
    /*
     * Interrupt driven signal from data1 data line
     */
    void data_1() {
        if (_pulses == W_PULSES) {
            clear();
        }
        _wiegand = ((_wiegand << 1) & 0x3ffffff) + 0x01;
        if (_pulses < W_PULSES / 2) {
            _o_par = (_o_par & 0x01) + 0x01;
        } else {
            _o_par = (_o_par & 0x01) + 0x01;
        }
        _pulses++;
    }
    
    /*
     * Various callable class members as to obtain data parts
     */
    unsigned long site() {
        return (_wiegand >> 17) & 0xff;
    }
    
    unsigned long data() {
        return (_wiegand >> 1) & 0xffff;
    }
    
    unsigned long rfid() {
        return (_wiegand >> 1) & 0xffffff;
    }
    
    /*
     * returns true if we have read all pulses and we have proper parity for both data halves
     */
    bool readable() {
        if (( _pulses == W_PULSES) && (_e_par == 0) && (_o_par == 1)) {
            return true;
        } else {
            return false;
        }
    }
    
    /*
     * clear last read card entry, prepare for next card entry
     */
    void clear() {
        _pulses = _e_par = _o_par = 0;
        _wiegand = 0;
    }
    
private:
    InterruptIn _D0;
    InterruptIn _D1;
    volatile unsigned long _wiegand;
    volatile unsigned int _pulses;
    volatile unsigned int _e_par;
    volatile unsigned int _o_par;
};