//---------------------------------------------------------------------------
// Copyright (C) 2000 Dallas Semiconductor Corporation, All Rights Reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included
// in all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
// OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY,  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
// IN NO EVENT SHALL DALLAS SEMICONDUCTOR BE LIABLE FOR ANY CLAIM, DAMAGES
// OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
// OTHER DEALINGS IN THE SOFTWARE.
//
// Except as contained in this notice, the name of Dallas Semiconductor
// shall not be used except as stated in the Dallas Semiconductor
// Branding Policy.
//---------------------------------------------------------------------------
//
//  TODO.C - COM functions required to communicate with the DS2480 based
//           Universal Serial Adapter 'U'.  Fill in the platform specific code.
//
//  Version: 2.00
//
//  History: 1.00 -> 1.01  Added function msDelay.
//
//           1.01 -> 1.02  Changed to generic OpenCOM/CloseCOM for easier
//                         use with other platforms.
//
//           1.02 -> 1.03  Removed caps in #includes for Linux capatibility
//                         Add function msGettick()
//           1.03 -> 2.00  Changed 'MLan' to 'ow'. Added support for
//                         multiple ports.
//           2.00 -> 2.10  some changes made as to allow MBED usage and multiple ports
//
//                         work to be done, evaluate functions at been removed / flagged
//                         as text if migrate to mbed is meaningfull / needed
//

//#include "onewire_config.h"   // missing include file, handled locally now

#include "ownet.h"
#include "mbed.h"
#include <string> 
using namespace std;

#define ONEWIRE_TX p9
#define ONEWIRE_RX p10

Serial com1( ONEWIRE_TX , ONEWIRE_RX );  // uart 1 = p13_p14, tx, rx 
Serial com2( ONEWIRE_TX , ONEWIRE_RX );  // uart 2 = p28_p27, tx, rx
Serial com3( ONEWIRE_TX , ONEWIRE_RX );  // uart 3 = p9_p10,  tx, rx

// exportable functions required
SMALLINT OpenCOM(int,char *);
SMALLINT WriteCOM(int,int,uchar *);
void     FlushCOM(int);
int      ReadCOM(int,int,uchar *);
void     BreakCOM(int);
void     SetBaudCOM(int,int);
void     CloseCOM(int);
long     msGettick(void);
void     msDelay(int);
SMALLINT owHasPowerDelivery(int);
SMALLINT owHasOverDrive(int);
SMALLINT owHasProgramPulse(int);
SMALLINT owWriteBytePower(int,SMALLINT);
SMALLINT owReadBitPower(int,SMALLINT);


//---------------------------------------------------------------------------
// Attempt to open a com port.  Keep the handle in ComID.
// Set the starting baud rate to 9600.
//
// 'portnum'   - number 0 to MAX_PORTNUM-1.  This number provided will
//               be used to indicate the port number desired when calling
//               all other functions in this library.
//
// 'port_zstr' - zero terminate port name.  For this platform
//               use format COMX where X is the port number.
//               NOTE: not being used by MBED
//
//
// Returns: TRUE(1)  - success, COM port opened
//          FALSE(0) - failure, could not open specified port
//
SMALLINT OpenCOM(int portnum, char *port_zstr) {

    switch (portnum) {
        case 1:
            com1.baud( 9600);
            break;
        case 2:
            com2.baud( 9600);
            break;
        case 3:
            com3.baud( 9600);
            break;
        default:
            return 0;
    }
    return 1;
}

//---------------------------------------------------------------------------
// Attempt to open a com port.  Keep the handle in ComID.
// Set the starting baud rate to 9600.
//
// 'portnum'   - number 0 to MAX_PORTNUM-1.  This number provided will
//               be used to indicate the port number desired when calling
//               all other functions in this library.
//
//
// Returns: the port number if it was successful otherwise -1
//
int OpenCOMEx(char *port_zstr) {

    int portnum = -1;

    if ( port_zstr[0] == '1') {
        portnum = 1;
    }
        
    if ( port_zstr[0] == '2') {
        portnum = 2;
    }

    if ( port_zstr[0] == '3') {
        portnum = 3;
    }    

    if(!OpenCOM(portnum, port_zstr)) {
        return -1;
    }
    return portnum;
}

//---------------------------------------------------------------------------
// Closes the connection to the port.
//
// 'portnum'  - number 0 to MAX_PORTNUM-1.  This number was provided to
//              OpenCOM to indicate the port number.
//
void CloseCOM(int portnum) {
   // no need to implement, mbed doesn't support close !
}

//---------------------------------------------------------------------------
// Flush the rx and tx buffers
//
// 'portnum'  - number 0 to MAX_PORTNUM-1.  This number was provided to
//              OpenCOM to indicate the port number.
//
void FlushCOM(int portnum) {

    switch (portnum) {
        case 1:
            // Flush TX & RX FIFO buffer
            LPC_UART1->FCR |= 0x06;
            break;
        case 2:
            // Flush TX & RX FIFO buffer
            LPC_UART2->FCR |= 0x06;
            break;
        case 3:
            // Flush TX & RX FIFO buffer
            LPC_UART3->FCR |= 0x06;
            break;
    }
}

//--------------------------------------------------------------------------
// Write an array of bytes to the COM port, verify that it was
// sent out.  Assume that baud rate has been set.
//
// 'portnum'  - number 0 to MAX_PORTNUM-1.  This number was provided to
//              OpenCOM to indicate the port number.
// 'outlen'   - number of bytes to write to COM port
// 'outbuf'   - pointer ot an array of bytes to write
//
// Returns:  TRUE(1)  - success
//           FALSE(0) - failure
//
SMALLINT WriteCOM(int portnum, int outlen, uchar *outbuf) {

    //printf("port-%d\r\n",portnum);
    for ( int x=0; x < outlen; x++ )
    {
        switch (portnum) {
            case 1:
                if (com1.writeable()) {
                    com1.putc( outbuf[x]);
                }
                break;
            case 2:
                if (com2.writeable()) {
                    com2.putc( outbuf[x]);
                }
                break;
            case 3:
                if (com3.writeable()) {
                    com3.putc( outbuf[x]);
                    //printf("idx-%d, char-%2x\r\n", x, outbuf[x]);
                }
                break;
        }
        wait_ms(1);
    }

    switch (portnum) {
        case 1:
            while ( (LPC_UART1->LSR & 0x40) == 0 ) {}
            break;
        case 2:
            while ( (LPC_UART2->LSR & 0x40) == 0 ) {}
            break;
        case 3:
            while ( (LPC_UART3->LSR & 0x40) == 0 ) {}
            break;
    }
    return 1;
}

//--------------------------------------------------------------------------
// Read an array of bytes from the COM port, verify that it was
// received.  Assume that baud rate has been set.
//
// 'portnum'  - number 0 to MAX_PORTNUM-1.  This number was provided to
//               OpenCOM to indicate the port number.
// 'inlen'     - number of bytes to read from COM port
// 'inbuf'     - pointer to a buffer to hold the incoming bytes
//
// Returns: number of characters read
//
int ReadCOM(int portnum, int inlen, unsigned char *inbuf) {

    string buffer ("");
    Timer t;
    int numchar = 0;
    int read =0;
    
    t.start();  // Start timeout timer

    while ( t.read_ms() < 1000 )  {             // If we haven't timed out....
        switch (portnum) {
            case 1:
                if (com1.readable()) {
                    read = com1.getc();
                    buffer += read;
                    numchar++;
                    t.reset(); 
                }
                break;
            case 2:
                if (com2.readable()) {
                    read = com2.getc();
                    buffer += read;
                    numchar++;
                    t.reset();
                }
                break;
            case 3:
                if (com3.readable()) {
                    read = com3.getc();
                    buffer += read;
                    numchar++;
                    t.reset();
                }
                break;
        }

        if (numchar == inlen)
        {
            break;  
        }
    }

    buffer.copy( (char*)inbuf, numchar );
   
    return numchar;
}

//--------------------------------------------------------------------------
// Send a break on the com port for at least 2 ms
//
// 'portnum'  - number 0 to MAX_PORTNUM-1.  This number was provided to
//               OpenCOM to indicate the port number.
//
void BreakCOM(int portnum) {
    int BREAK_time = 2;

    switch (portnum) {
        case 1:
            // Send Line Break
            LPC_UART1->LCR |= 0x40;
            wait_ms( BREAK_time );
            // Mark after break        
            LPC_UART1->LCR &= ~(0x40);
            break;
        case 2:
            // Send Line Break
            LPC_UART2->LCR |= 0x40;
            wait_ms( BREAK_time );
            // Mark after break        
            LPC_UART2->LCR &= ~(0x40);
            break;
        case 3:
            // Send Line Break
            LPC_UART3->LCR |= 0x40;
            wait_ms( BREAK_time );   
            // Mark after break        
            LPC_UART3->LCR &= ~(0x40);
            break;
        default:
            break;
    }
    
    wait_us( BREAK_time );
}

//--------------------------------------------------------------------------
// Set the baud rate on the com port.
//
// 'portnum'   - number 0 to MAX_PORTNUM-1.  This number was provided to
//               OpenCOM to indicate the port number.
// 'new_baud'  - new baud rate defined as
//                PARMSET_9600     0x00
//                PARMSET_19200    0x02
//                PARMSET_57600    0x04
//                PARMSET_115200   0x06
//
void SetBaudCOM(int portnum, int new_baud) {

    int baud_rate = 9600;
    
    switch (new_baud) {
        case 0:
            baud_rate = 9600;
            break;
        case 2:
            baud_rate = 19200;
            break;
        case 4:
            baud_rate = 57600;
            break;
         case 6:
            baud_rate = 115200;
            break;
    }

    switch (portnum) {

        case 1:
            com1.baud(baud_rate);
            break;
        case 2:
            com2.baud(baud_rate);
            break;
        case 3:
            com2.baud(baud_rate);
            break;
    }
}

//--------------------------------------------------------------------------
//  Description:
//     Delay for at least 'len' ms
//
void msDelay(int len)
{
    wait_ms( len );
}

//--------------------------------------------------------------------------
// Get the current millisecond tick count.  Does not have to represent
// an actual time, it just needs to be an incrementing timer.
//
long msGettick(void) {
   // add platform specific code here
   return 0;
}

/* from here onwards functions have been removed, further evaluation needed
//--------------------------------------------------------------------------
// This procedure indicates wether the adapter can deliver power.
//
// 'portnum'  - number 0 to MAX_PORTNUM-1.  This number was provided to
//              OpenCOM to indicate the port number.
//
// Returns:  TRUE  if adapter is capable of delivering power.
//
SMALLINT owHasPowerDelivery(int portnum) {

   // add adapter specific code here
   return TRUE;
}

//--------------------------------------------------------------------------
// This procedure indicates wether the adapter can deliver power.
//
// 'portnum'  - number 0 to MAX_PORTNUM-1.  This number was provided to
//              OpenCOM to indicate the port number.
//
// Returns:  TRUE  if adapter is capable of over drive.
//
SMALLINT owHasOverDrive(int portnum) {

   // add adapter specific code here
   return TRUE;
}

//--------------------------------------------------------------------------
// This procedure creates a fixed 480 microseconds 12 volt pulse
// on the 1-Wire Net for programming EPROM iButtons.
//
// 'portnum'  - number 0 to MAX_PORTNUM-1.  This number was provided to
//              OpenCOM to indicate the port number.
//
// Returns:  TRUE  program volatage available
//           FALSE program voltage not available
SMALLINT owHasProgramPulse(int portnum)
{
   // add adapter specific code here
   return TRUE;
}


//--------------------------------------------------------------------------
// Send 8 bits of communication to the 1-Wire Net and verify that the
// 8 bits read from the 1-Wire Net is the same (write operation).
// The parameter 'sendbyte' least significant 8 bits are used.  After the
// 8 bits are sent change the level of the 1-Wire net.
//
// 'portnum'  - number 0 to MAX_PORTNUM-1.  This number was provided to
//              OpenCOM to indicate the port number.
// 'sendbyte' - 8 bits to send (least significant byte)
//
// Returns:  TRUE: bytes written and echo was the same
//           FALSE: echo was not the same
//
SMALLINT owWriteBytePower(int portnum, SMALLINT sendbyte)
{
   // replace if platform has better implementation (faster response)
   if (!owHasPowerDelivery(portnum))
      return FALSE;

   if(owTouchByte(portnum,sendbyte) != sendbyte)
      return FALSE;

   if(owLevel(portnum,MODE_STRONG5) != MODE_STRONG5)
      return FALSE;

   return TRUE;
}


//--------------------------------------------------------------------------
// Send 1 bit of communication to the 1-Wire Net and verify that the
// response matches the 'applyPowerResponse' bit and apply power delivery
// to the 1-Wire net.  Note that some implementations may apply the power
// first and then turn it off if the response is incorrect.
//
// 'portnum'  - number 0 to MAX_PORTNUM-1.  This number was provided to
//              OpenCOM to indicate the port number.
// 'applyPowerResponse' - 1 bit response to check, if correct then start
//                        power delivery
//
// Returns:  TRUE: bit written and response correct, strong pullup now on
//           FALSE: response incorrect
//
SMALLINT owReadBitPower(int portnum, SMALLINT applyPowerResponse)
{
   // replace if platform has better implementation (faster response)
   if (!owHasPowerDelivery(portnum))
      return FALSE;

   if(owTouchBit(portnum,0x01) != applyPowerResponse)
      return FALSE;

   if(owLevel(portnum,MODE_STRONG5) != MODE_STRONG5)
      return FALSE;

   return TRUE;
}

*/