/* mbed Microcontroller Library
 * Copyright (c) 2018 ARM Limited
 * SPDX-License-Identifier: Apache-2.0
 */

#include <mbed.h>

#define CH_INIT             0
#define CH_BOOST_START      1
#define CH_BOOSTING         2
#define CH_POST_BOOST       3
#define CH_WAIT_200MS       4
#define CH_WAIT_800MS       5

#define SAMPLE_FLAG1 (1UL << 0)

Serial pc(p6, p8);

DigitalOut led1(LED1);  // Used to indicate boost state to user
DigitalOut led2(LED2);  // Used to indicate boost state to user

DigitalOut ch1(LED4);

Timeout chStateTimeout;

Timer t;

Thread thread;

//Semaphore sem(0);

EventFlags event_flags;

volatile bool ch_progress = false;
volatile bool ch_boost_done = false;

// -----------------------------------------------------------------------------
void timeoutHandler() {
    ch_progress = true;
}

// -----------------------------------------------------------------------------
void chBoostThread() {
    while (1) {
        //sem.wait();
        //Thread::signal_wait(0x1);
        event_flags.wait_any(SAMPLE_FLAG1);
        
        t.start();
        ch1 = 1;
        wait_us(100);
        ch1 = 0;
        t.stop();
        ch_boost_done = true;        
    }
}

// -----------------------------------------------------------------------------
int main()
{
    uint8_t ch_state = CH_INIT;
    unsigned long dummy_count = 0, dummy_count_post = 0;
    
    ch1 = 0;
    
    thread.start(chBoostThread);    
    
    while(1) {
        switch (ch_state) {
            case CH_INIT:
                ch1 = 0;
                ch_progress = false;
                ch_boost_done = false;
                dummy_count = 0;
                dummy_count_post = 0;
                ch_state = CH_BOOST_START;
                break;
            case CH_BOOST_START:          
                ch_state = CH_BOOSTING;
                led1 = 1;   // Give visual indication at start of boost stage
                //sem.release();
                //thread.signal_set(0x1);
                event_flags.set(SAMPLE_FLAG1);
            case CH_BOOSTING:
                dummy_count++;  // measure response during 100us boost stage
                
                if (ch_boost_done) {
                    ch_state = CH_POST_BOOST;
                    chStateTimeout.attach_us(&timeoutHandler, 1000);
                }
                else break;
            case CH_POST_BOOST:
                dummy_count_post++;  // measuring response after boost stage for 1ms
                
                if (ch_progress) {
                    ch_progress = false;
                    ch_state = CH_WAIT_200MS;
                    chStateTimeout.attach(&timeoutHandler, 0.2f);
                }
                break;
            case CH_WAIT_200MS:
                if (ch_progress) {
                    ch_progress = false;
                    led1 = 0;   // Turn off boost indication
                    ch_state = CH_WAIT_800MS;
                    chStateTimeout.attach(&timeoutHandler, 0.8f);
                }
                break;
            case CH_WAIT_800MS:
                if (ch_progress) {
                    ch_progress = false;
                    ch_state = CH_INIT;
                    
                    pc.printf("CH_BOOSTING COUNT: %lu; CH_POST_BOOST COUNT: %lu; TIME TAKEN: %d\r\n", dummy_count, dummy_count_post, t.read_us());    // print measured response;
                    t.reset();
                }
                break;            
        }
        
    } 
}
