/**
 ******************************************************************************
 * @file
 * @author  Paul Paterson
 * @version
 * @date    2015-12-14
 * @brief   CANOpen implementation library
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2015 Paul Paterson
 *
 * All rights reserved.

 This program is free software: you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, either version 3 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PPCAN_NODE_PIN0808_H
#define PPCAN_NODE_PIN0808_H

#include "Node.h"

namespace ppCANOpen
{
    
/** Custom Object Dictionary for pinball io_0808 device.
  */
class Node_pin0808 : public Node
{
    
public:
    Node_pin0808(int id, ServiceProvider * provider, int bLoop = 0);
    virtual ~Node_pin0808(void);

private:
    /* constants */
    /** milliseconds necessary to debounce all input */
    static const uint8_t DEBOUNCE_TIME = 3;

    /* IO details */
    uint8_t prevInputBuffers[1];
    uint8_t inputDebounce[8];    
    
    uint8_t prevOutputBuffers[1];
    int scheduleIndexTime; 

    /**************************************************************************
     * Implementation Overrides 
     **************************************************************************
     */

    /* Application run */
    virtual void OnFixedUpdate(void);
    virtual void OnUpdate(void);

    /* SYNC */
    virtual void OnSync (uint8_t counter);

    /* NMT Control */  
    virtual void OnInitialize (void);
    virtual void OnPreoperational (void);
    virtual void OnOperational (void);
    virtual void OnStopped (void);
    
    
    /**************************************************************************
     * Object Dictionary 
     **************************************************************************
     */
    
    /* Communication --------------------------------------------------------*/
    /* index 0x1200 :   SDO Server */
    ObjectData      Obj1200;
    SubIndexSize    Obj1200_highestSubIndex;
    uint32_t        Obj1200_ReceiveCobId;
    uint32_t        Obj1200_TransmitCobId;
    EntryData       Obj1200_entries[3];
    
    /* index 0x1400 :   Receive PDO 1 Parameter (Write Digital Output) */
    ObjectData      Obj1400;
    SubIndexSize    Obj1400_highestSubIndex;
    uint32_t        Obj1400_CobId;
    uint8_t         Obj1400_TransmissionType;
    uint16_t        Obj1400_InhibitTime;
    uint8_t         Obj1400_CompatibilityEntry;
    uint16_t        Obj1400_EventTimer;
    EntryData       Obj1400_entries[6];
    
    /* index 0x1401 :   Receive PDO 2 Parameter (Write Output Configuration) */
    ObjectData      Obj1401;
    SubIndexSize    Obj1401_highestSubIndex;
    uint32_t        Obj1401_CobId;
    uint8_t         Obj1401_TransmissionType;
    uint16_t        Obj1401_InhibitTime;
    uint8_t         Obj1401_CompatibilityEntry;
    uint16_t        Obj1401_EventTimer;
    EntryData       Obj1401_entries[6];
    
    /* index 0x1402 :   Receive PDO 3 Parameter (Write/Clear Autotrigger Rule) */
    ObjectData      Obj1402;
    SubIndexSize    Obj1402_highestSubIndex;
    uint32_t        Obj1402_CobId;
    uint8_t         Obj1402_TransmissionType;
    uint16_t        Obj1402_InhibitTime;
    uint8_t         Obj1402_CompatibilityEntry;
    uint16_t        Obj1402_EventTimer;
    EntryData       Obj1402_entries[6];
    
    /* index 0x1403 :   Receive PDO 5 Parameter (input changes) */
    ObjectData      Obj1403;
    SubIndexSize    Obj1403_highestSubIndex;
    uint32_t        Obj1403_CobId;
    uint8_t         Obj1403_TransmissionType;
    uint16_t        Obj1403_InhibitTime;
    uint8_t         Obj1403_CompatibilityEntry;
    uint16_t        Obj1403_EventTimer;
    EntryData       Obj1403_entries[6];
    
    /* index 0x1600 :   Receive PDO 1 Mapping */
    ObjectData      Obj1600;
    SubIndexSize    Obj1600_highestSubIndex;
    uint32_t        Obj1600_Map;
    EntryData       Obj1600_entries[2];
    
    /* index 0x1601 :   Transmit PDO 1 Mapping */
    ObjectData      Obj1601;
    SubIndexSize    Obj1601_highestSubIndex;
    uint32_t        Obj1601_Map;
    EntryData       Obj1601_entries[2];
    
    /* index 0x1602 :   Transmit PDO 1 Mapping */
    ObjectData      Obj1602;
    SubIndexSize    Obj1602_highestSubIndex;
    uint32_t        Obj1602_Map;
    EntryData       Obj1602_entries[2];
    
    /* index 0x1603 :   Receive PDO 2 Mapping */
    ObjectData      Obj1603;
    SubIndexSize    Obj1603_highestSubIndex;
    uint32_t        Obj1603_Map;
    EntryData       Obj1603_entries[2];
    
    
    /* index 0x1800 :   Transmit PDO 1 Parameter */
    ObjectData      Obj1800;
    SubIndexSize    Obj1800_highestSubIndex;
    uint32_t        Obj1800_CobId;
    uint8_t         Obj1800_TransmissionType;
    uint16_t        Obj1800_InhibitTime;
    uint8_t         Obj1800_CompatibilityEntry;
    uint16_t        Obj1800_EventTimer;
    EntryData       Obj1800_entries[6];
    
    /* index 0x1A00 :   Transmit PDO 1 Mapping */
    ObjectData      Obj1A00;
    SubIndexSize    Obj1A00_highestSubIndex;
    uint32_t        Obj1A00_MapInput;
    uint32_t        Obj1A00_MapVoid16;
    uint32_t        Obj1A00_MapChange;
    uint32_t        Obj1A00_MapSourceId;
    EntryData       Obj1A00_entries[6];
    
    
    /* Manufacturer Specific ------------------------------------------------*/
    /* index 0x2001 : Pin Output Configurations */
    struct OutputConfiguration {
        static const uint8_t PULSE          = 0x01;
        static const uint8_t PATTER         = 0x02;
        static const uint8_t PULSED_PATTER  = 0x04;
        //static const uint8_t SCHEDULED      = 0x08;
          
        uint8_t     type;
        uint8_t     pulse_ms;
        uint8_t     pwm_on;
        uint8_t     pwm_off;
        uint32_t    writeData;
    };
    
    ObjectData          Obj2001;
    SubIndexSize        Obj2001_highestSubIndex;
    OutputConfiguration writeOutputConfig;
    OutputConfiguration outputConfigs[8];
    OutputConfiguration outputTimers[8]; /* not actually part of dictionary */
    EntryData           Obj2001_entries[10];
    
    /* index 0x2002 : Output Schedule Configuration */
    ObjectData      Obj2002;
    SubIndexSize    Obj2002_highestSubIndex;
    uint32_t        scheduleConfig;
    uint32_t        schedules[8];  
    EntryData       Obj2002_entries[10];
    
    /* index 0x2100 : Input change data */
    ObjectData      Obj2100;
    SubIndexSize    Obj2100_highestSubIndex;
    uint8_t         inputChangeMask[1];  
    /* uint8_t         nodeId; */
    EntryData       Obj2100_entries[3];
        
    /* index 0x2200 : Autotrigger Rules */
    struct AutotriggerRule {
        uint8_t     sourceId;
        uint8_t     input; /* 0-4: inputNum (0-23)
                              5-7: 0 on inactive
                                   1 on active
                                   2 on any change
                                   3 ???
                            */
        uint8_t     setMask[3];
        uint8_t     clearMask[3];  
    };
    
    struct AutotriggerMessage {
        uint8_t     input[3];
        uint8_t     change[3];
        uint8_t     sourceId;
    };
    
    ObjectData          Obj2200;
    SubIndexSize        Obj2200_highestSubIndex;
    AutotriggerRule     writeRule;
    AutotriggerMessage  autotriggerMessage;
    AutotriggerRule     rules[24];
    EntryData           Obj2200_entries[27];

    
    /* Device Specific ------------------------------------------------------*/
    /* index 0x6000 :   Mapped variable "inputs" */
    ObjectData      Obj6000;
    SubIndexSize    Obj6000_highestSubIndex; 
    uint8_t         readInputBuffers[1];
    EntryData       Obj6000_entries[2];
    
    /* index 0x6005 :   Global Input Interrupt Enable */
    ObjectData      Obj6005;
    SubIndexSize    Obj6005_highestSubIndex; 
    uint8_t         bInputInterruptEnable;
    EntryData       Obj6005_entries[2];
    
    /* index 0x6006 :   Any Change Interrupt Mask */
    ObjectData      Obj6006;
    SubIndexSize    Obj6006_highestSubIndex; 
    uint8_t         inputInterruptMask[1];
    EntryData       Obj6006_entries[2];
    
    /* index 0x6200 :   Mapped variable "outputs" */
    ObjectData      Obj6200;
    SubIndexSize    Obj6200_highestSubIndex; 
    uint8_t         writeOutputBuffers[1];
    EntryData       Obj6200_entries[2];
    
    /* Scan Method */
    virtual ObjectData * ScanIndex(IndexSize index);
};

} /* namespace ppCANOpen */

#endif // PPCAN_NODE_PIN0808_H
