/**
 ******************************************************************************
 * @file
 * @author  Paul Paterson
 * @version
 * @date    2015-12-14
 * @brief   CANOpen implementation library
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2015 Paul Paterson
 *
 * All rights reserved.

 This program is free software: you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, either version 3 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PPCAN_NODE_H
#define PPCAN_NODE_H

#include "ObjectDictionary.h"

/*=========================================================================
 * Forward declarations
 *=========================================================================
 */
struct CanOpenMessage;
//struct ObjectData;

/* NMT Constants --------------------------------------------------------*/
#define NMT_STATE_TOGGLE_BIT      0x80

/* NMT Macros -----------------------------------------------------------*/
#define NMT_SET_STATE(curState, newState) (curState) = (((curState) & NMT_STATE_TOGGLE_BIT) | ((newState) & (~NMT_STATE_TOGGLE_BIT))))

namespace ppCANOpen
{

/* Avoid circular reference */
class ServiceProvider;


/** Node Class to implement feature of a CANOpen NMT node
 */
class Node
{
public:

    /* ========================================================================
     * Internal structures and constants
     * ========================================================================
     */
     
    /** Node network management state */
    struct State {
        static const int INITIALIZED     = 0x00;
        static const int DISCONNECTED    = 0x01;
        static const int CONNECTING      = 0x02;
        static const int PREPARING       = 0x02;
        static const int STOPPED         = 0x04;
        static const int OPERATIONAL     = 0x05;
        static const int PREOPERATIONAL  = 0x7F;
        static const int UNKNOWN         = 0x0F;
        
        int nmtState;
        int bBoot;
        int bSDO;
        int bEmergency;
        int bSYNC;
        int bLifeGuard;
        int bPDO;
        int bLSS;
        
        int bLifeGuardToggle;
    };

    /* ========================================================================
     * Construction
     * ========================================================================
     */
     
    /** Maintain the multitude of states for a node */
    Node (int id, ServiceProvider * provider, int bLoop = 0);    

    /* ========================================================================
     * Public Methods
     * ========================================================================
     */
    
    /** Call from ServiceProvider on an interrupt */
    void FixedUpdate (uint32_t time);
    
    /** Call from ServiceProvider every loop of Run() */
    void Update (void);
    
    /** Handle message given by the ServiceProvider*/
     int DispatchMessage(CanOpenMessage *canOpenMsg);
     
    /* ========================================================================
     * Public Member Variables
     * ========================================================================
     */
     
    /* Common Node properties -----------------------------------------------*/
    /** Network id for the node
      * @note Ref is held in object dictionary and upon setting, may need to
      * update some other objects.
      */
    int nodeId;

    /** Loopback Mode.  Default off (0), if on (1) then will hear it's own 
      * messages
      */
    int bLoopbackOn;

protected:

    /* ========================================================================
     * Protected? Methods to handle various messages
     * Used to deal with everything from inside of Dispatch Message
     * ========================================================================
     */
    
    /* PDO (7.2.2), MPDO (7.2.3) --------------------------------------------*/
    
    int HandlePdo (CanOpenMessage *canOpenMsg);
    
    
    /* SDO (7.2.4) ----------------------------------------------------------*/
    int HandleSdo (CanOpenMessage *canOpenMsg);
    
    int HandleExpeditedDownload (int sdoNum, int index, int subIndex, int dataCount, char * data);
    int HandleInitiateDownloadRequest (int sdoNum, int index, int subIndex, int dataCount, char * data);
    
    int HandleExpeditedUpload (int sdoNum, int index, int subIndex);
    
    
    // TODO: express and not express
    
    /* SYNC object (7.2.5) --------------------------------------------------*/
    int HandleSync (CanOpenMessage *canOpenMsg);
    
    /* Time Stamp object (7.2.6) --------------------------------------------*/
    /* Handled all inside of the ServiceProvider */
    
    /* Emergency object (7.2.7) ---------------------------------------------*/
    
    int ConsumeEmergency (void);// TODO: fix params
    
    
    /* Network Management (7.2.8) -------------------------------------------*/
    /* ---- Node Control (7.2.8.2.1) ----------------------------------------*/
    int HandleNodeControl (CanOpenMessage *canOpenMsg);


    /* ---- Error Control (7.2.8.2.2) ---------------------------------------*/
    
    int HandleNodeGuardRequest (int masterId);
    int ConsumeHeartbeat (int producerId);
    
protected:

    /* ========================================================================
     * Protected Methods to implement node application in derived classes
     * ========================================================================
     */
     
     /** Perform actions every cycle (when in operational mode)
       * @note Override to implement user application
       * @note The OnTick function should avoid modifying data in the object
       * dictionary unless necessary, because there will almost certainly be 
       * a race condition.
       */
     virtual void OnFixedUpdate(void) = 0;
     
     
     
     virtual void OnUpdate(void) = 0;
    
    
    /* SYNC -----------------------------------------------------------------*/
    /** Perform actions when state changed to stop
      * @note Override to implement user application
      */
    virtual void OnSync (uint8_t counter) = 0;
    
    
    /* NMT Node Control -----------------------------------------------------*/
    /** Perform actions when node reset
      * @note Override to implement user application
      */
    virtual void OnInitialize (void) = 0;
    
    /** Perform actions when state changed to pre-operational
      * @note Override to implement user application
      */
    virtual void OnPreoperational (void) = 0;
    
    /** Perform actions when state changed to operational
      * @note Override to implement user application
      */
    virtual void OnOperational (void) = 0;
    
    /** Perform actions when state changed to stop
      * @note Override to implement user application
      */
    virtual void OnStopped (void) = 0;
    
    /* Object Dictionary Handling -------------------------------------------*/
    /** Abstract method to give access to the object entries of derived
      * classes
      */
    virtual ObjectData * ScanIndex(IndexSize index) = 0;


    /* ========================================================================
     * Protected Methods to Post Messages
     * ========================================================================
     */
     
    int PostTPDO (int cobId);


    /* ========================================================================
     * Protected Member Variables
     * ========================================================================
     */
     
    /* Object Dictionary Handling -------------------------------------------*/
    /** Array of ObjectData to contain all of the object dictionary data */
    ObjectData *dictionary;
    
    /* Elapsed Time ---------------------------------------------------------*/
    /** Millisecond time stamp at this OnTick() */
    uint32_t timeCurrentTick;
    
    /** Millisecond difference between current OnTick() and the previous */
    uint32_t timeSinceLastTick;

// should be private
// TODO: implement functions in Node class to post all types of messages so
//       derived classes do not have to consider message structure.
protected:

    /* ========================================================================
     * Private members
     * ========================================================================
     */
     
    /* ServiceProvider ------------------------------------------------------*/
    /** Reference to Service Provider that this node is attached to */
    ServiceProvider * pMyProvider;

private:  
    /* Common Node Methods --------------------------------------------------*/
    /** Change state of the node */
    void ChangeState(int newState);
    
    /* Common Node properties -----------------------------------------------*/    
    /** Network and communication state of the node */
    State state;
    
    
};

} /* namespace ppCANOpen */

#endif // PPCAN_NODE_H

