/**
 ******************************************************************************
 * @file
 * @author  Paul Paterson
 * @version
 * @date    2015-12-14
 * @brief   CANOpen implementation library
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2015 Paul Paterson
 *
 * All rights reserved.

 This program is free software: you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, either version 3 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PPCAN_CANOPEN_MESSAGE_H
#define PPCAN_CANOPEN_MESSAGE_H

#include "stdint.h"

#ifdef __cplusplus
extern "C" {
#endif

/*=========================================================================
 * CANOpen Message Structure
 *=========================================================================
 */

/** CANOpen Message Format */
typedef enum {
    CANOPEN_FORMAT_STANDARD = 0,
    CANOPEN_FORMAT_EXTENDED,
    CANOPEN_FORMAT_ANY
} CanOpenFormat;

/** CANOpen Message Data Type */
typedef enum {
    CANOPEN_TYPE_DATA = 0,
    CANOPEN_TYPE_REMOTE
} CanOpenType;


/** CANOpen Message */
typedef struct CanOpenMessage {
    uint32_t        id;
    uint8_t         data[8];
    CanOpenFormat   format;
    CanOpenType     type;
    uint8_t         dataCount;
} CanOpenMessage;


/*=========================================================================
 * CANOpen Function Codes
 *=========================================================================
 */
 
/** CANOpen Function Codes */
typedef enum {
    CANOPEN_FUNCTION_CODE_NMT        = 0x00,
    CANOPEN_FUNCTION_CODE_SYNC       = 0x01,
    CANOPEN_FUNCTION_CODE_TIME       = 0x02,
    CANOPEN_FUNCTION_CODE_PDO1T      = 0x03,
    CANOPEN_FUNCTION_CODE_PDO1R      = 0x04,
    CANOPEN_FUNCTION_CODE_PDO2T      = 0x05,
    CANOPEN_FUNCTION_CODE_PDO2R      = 0x06,
    CANOPEN_FUNCTION_CODE_PDO3T      = 0x07,
    CANOPEN_FUNCTION_CODE_PDO3R      = 0x08,
    CANOPEN_FUNCTION_CODE_PDO4T      = 0x09,
    CANOPEN_FUNCTION_CODE_PDO4R      = 0x0A,
    CANOPEN_FUNCTION_CODE_SDOT       = 0x0B,
    CANOPEN_FUNCTION_CODE_SDOR       = 0x0C,
    CANOPEN_FUNCTION_CODE_NODE_GUARD = 0x0E,
    CANOPEN_FUNCTION_CODE_LSS        = 0x0F
} CanOpenFunctionCodes;

/* Message Constants */
#define MESSAGE_NODEID_BITS     0b00001111111
#define MESSAGE_COMMAND_BITS    0b11110000000

/* Message Macros -----------------------------------------------------------*/ 
#define MESSAGE_GET_NODEID(cobId)   (cobId & MESSAGE_NODEID_BITS)
#define MESSAGE_GET_COMMAND(cobId)  ((cobId & MESSAGE_COMMAND_BITS) >> 7)

/*=========================================================================
 * SDO MESSAGE PARAMETERS
 *=========================================================================
 */

/** SDO initiate protocol command specifiers */
typedef enum {
     SDO_CCS_DOWNLOAD_SEGMENT_REQUEST   = 0x00, 
     SDO_CCS_INITIATE_DOWNLOAD_REQUEST  = 0x01,
     SDO_CCS_INITIATE_UPLOAD_REQUEST    = 0x02,
} SdoClientCommandSpecifier;

/** SDO segment protocol command specifiers */
typedef enum {
     SDO_SCS_DOWNLOAD_SEGMENT_RESPONSE  = 0x01,
     SDO_SCS_INITIATE_UPLOAD_RESPONSE   = 0x02,
     SDO_SCS_INITIATE_DOWNLOAD_RESPONSE = 0x03,
} SdoServerCommandSpecifier;

/* SDO constants --------------------------------------------------------*/
#define SDO_SIZE_INDICATOR_BIT  0b00000001
#define SDO_TRANSFER_TYPE_BIT   0b00000010
#define SDO_DATA_COUNT_BITS     0b00001100
#define SDO_TOGGLE_BIT          0b00010000
#define SDO_CS_BITS             0b11100000

/* SDO macros -----------------------------------------------------------*/ 
#define SDO_GET_CS(data0)           ((data0 & SDO_CS_BITS) >> 5)
#define SDO_GET_DATA_COUNT(data0)   ((data0 & SDO_DATA_COUNT_BITS) >> 2)


/*=========================================================================
 * NMT MESSAGE PARAMETERS
 *=========================================================================
 */

/** NMT node control protocol command specifiers */
typedef enum {
     NMT_CS_START       = 0x01,
     NMT_CS_STOP        = 0x02,
     NMT_CS_ENTER_PREOP = 0x80,
     NMT_CS_RESET_NODE  = 0x81,
     NMT_CS_RESET_COM   = 0x82
} NmtCommandSpecifier;

#ifdef __cplusplus
};
#endif

#endif /* PPCAN_CANOPEN_MESSAGE_H */
