/**
 ******************************************************************************
 * @file
 * @author  Paul Paterson
 * @version
 * @date    2015-12-22
 * @brief   CANOpen implementation library
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2015 Paul Paterson
 *
 * All rights reserved.

 This program is free software: you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, either version 3 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PPCAN_SERVICE_PROVIDER_H
#define PPCAN_SERVICE_PROVIDER_H

#include "CanOpenMessage.h"
#include <queue>

/*=========================================================================
 * Forward declarations
 *=========================================================================
 */  
 
struct CanOpenHandle;

namespace ppCANOpen
{

/* Avoid circular reference */
class Node;

/** Node Class to implement feature of a CANOpen NMT node
 */
class ServiceProvider
{
public:

    ServiceProvider (void);
    ~ServiceProvider (void);
    
    /* ========================================================================
     * Public Methods
     * ========================================================================
     */
    
    /* Main application -----------------------------------------------------*/
    /** Main loop */
    void Run (void);
    
    /** Iterate through Nodes and update */
    void UpdateNodes(void);
    
    /** C helper function to pass to API */
    static void UpdateNodes_CWrapper(void *pServiceObject);
    
    /* Node Management ------------------------------------------------------*/
    /** Register a node to get messages and get update calls */
    int AddNode (Node * node);
    
    // TODO: remove node (swap pointers to fill in nicely
    
      
    /* CanMessage Transmission ----------------------------------------------*/
    /** Add a message to the message queue outbox */
    void PostMessage (int nodeId, CanOpenMessage * msg);
    
    /** quick way to create and send an NMT Control message
      * @note Make posting a message easier! 
      */
    void PostNmtControl (char targetNodeId, NmtCommandSpecifier cs);
    
    /* Elapsed Time ---------------------------------------------------------*/
    /** A way for the the Nodes to get elapsed time when they want it */
    uint32_t GetTime(void);
    
private:
    /* ========================================================================
     * Private Constants
     * ========================================================================
     */
    
    /** Define size of arrays in one place */
    static const int SERVICE_MAX_NODES = 8;    

    /* ========================================================================
     * Private Methods
     * ========================================================================
     */

    /* CanMessage Reading ---------------------------------------------------*/
    /** Function to be called on every CAN read interrupt */
    void ReadIT(void);
    
    /** C helper function used to pass ReadIT() to API */
    static void ReadIT_CWrapper(void *pServiceObject);
    
    /** Helper function that utilizes backup message inbox when the main one
      * is locked by the application.
      */
    void ReadIT_clearBuffer(void);

    /* Elapsed Time ---------------------------------------------------------*/
    /** Synchronize elapsed time*/
    void HandleTimeMessage(CanOpenMessage *canOpenMsg); 
    
    
    /* ========================================================================
     * Private Member Variables
     * ========================================================================
     */
     
    /* Node Management -------------------------------------------------------*/
    /** array of nodes to cycle through. */
    Node * nodes[SERVICE_MAX_NODES];
    
    /** Keeps count of current number of nodes. */
    int nodeCount;    
    
    /* CanMessage Reading ---------------------------------------------------*/
    /** array of messages in a queue */
    std::queue<CanOpenMessage> inbox;
    
    /** Extra buffer only handled by ISR. Used if inbox is locked */
    std::queue<CanOpenMessage> inboxBuffer; 
    
    /** boolean to lock and unlock inbox  */
    int bInboxUnlocked;
    
    /** Data to keep track of internal messages.  Node ID allows SP to avoid
      * dispatching message to the one who sent it
      */
    struct InternalMessage {
        int             nodeId;
        CanOpenMessage  message;
        
        InternalMessage(uint8_t n, CanOpenMessage m):
            nodeId(n),
            message(m)
        {}
    };

    /** array of messages in a queue */
    std::queue<InternalMessage> outbox; 
    
    /* Elapsed Time ---------------------------------------------------------*/   
    /** Elapsed time offset used for synchronization*/
    int32_t elapsedTimeOffset; 
    
};    
    
} /* namspace ppCANOpen */

#endif /* PPCAN_SERVICE_PROVIDER_H */

