/**
 ******************************************************************************
 * @file
 * @author  Paul Paterson
 * @version
 * @date    2015-12-14
 * @brief   CANOpen implementation library
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2015 Paul Paterson
 *
 * All rights reserved.

 This program is free software: you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, either version 3 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PPCAN_OBJECT_DICTIONARY_H
#define PPCAN_OBJECT_DICTIONARY_H

#include <stdint.h>

namespace ppCANOpen
{

/** Data Type alias to provide context when defining the object library */
typedef uint8_t DataType;

/** Data Property alias to provide context when defining the object library */
typedef uint8_t DataProperty;

/** defines the data in a single entry of data (subindex)
 */
struct EntryData
{
    /* Data value type constants ----------------------------------------------
     * Taken from the CANOpen Standard.  they are part of the object dictionary
     * at indices 0x0001 to 0x0023.  Given to every Sub Index.
     */
     
    static const DataType TYPE_BOOLEAN                     = 0x01;
    static const DataType TYPE_INT8                        = 0x02;
    static const DataType TYPE_INT16                       = 0x03;
    static const DataType TYPE_INT32                       = 0x04;
    static const DataType TYPE_UINT8                       = 0x05;
    static const DataType TYPE_UINT16                      = 0x06;
    static const DataType TYPE_UINT32                      = 0x07;
    static const DataType TYPE_REAL32                      = 0x08;
    static const DataType TYPE_VISIBLE_STRING              = 0x09;
    static const DataType TYPE_OCTET_STRING                = 0x0A;
    static const DataType TYPE_UNICODE_STRING              = 0x0B;
    static const DataType TYPE_TIME_OF_DAY                 = 0x0C;
    static const DataType TYPE_TIME_DIFFERENCE             = 0x0D;
    /* RESERVED */
    static const DataType TYPE_DOMAIN                      = 0x0F;
    static const DataType TYPE_INT24                       = 0x10;
    static const DataType TYPE_REAL64                      = 0x11;
    static const DataType TYPE_INT40                       = 0x12;
    static const DataType TYPE_INT48                       = 0x13;
    static const DataType TYPE_INT56                       = 0x14;
    static const DataType TYPE_INT64                       = 0x15;
    static const DataType TYPE_UINT24                      = 0x16;
    /* RESERVED */
    static const DataType TYPE_UINT40                      = 0x18;
    static const DataType TYPE_UINT48                      = 0x19;
    static const DataType TYPE_UINT56                      = 0x1A;
    static const DataType TYPE_UINT64                      = 0x1B;
    /* RESERVED */
    static const DataType TYPE_PDO_COMMUNICATION_PARAMETER = 0x20;
    static const DataType TYPE_PDO_MAPPING                 = 0x21;
    static const DataType TYPE_SDO_COMMUNICATION_PARAMETER = 0x22;
    static const DataType TYPE_IDENTITY                    = 0x23;


    /* Data properties constants ----------------------------------------------  
     * Read/Write priveledges, save on powerdown, etc.
     * Can be strung together into a mask.
     */
     
    static const DataProperty PROPERTY_READABLE         = 0x01;
    static const DataProperty PROPERTY_WRITEABLE        = 0x02;
    static const DataProperty PROPERTY_READ_WRITEABLE   = 0x03;
    /* static const uint8_t PROPERTY_STATIC                = 0x04 *//* possible to save setting after powerdown */
    
    /* EntryData data --------------------------------------------------------
     */
    
    void           *pData;
    uint16_t        size;
    DataType        type;
    DataProperty    properties;
    
    EntryData(){}
    
    EntryData(void *d, uint16_t s, DataType t, DataProperty p)
    : pData(d), size(s), type(t), properties(p)
    {}
};


/** Index alias to provide context when defining the object library */
typedef uint16_t IndexSize;

/** Sub-Index alias to provide context when defining the object library */
typedef uint8_t SubIndexSize;

/** defines the data for a single index, including an array of subindices
 */
struct ObjectData
{                
    EntryData       *entries;
    IndexSize        index;
    SubIndexSize     entryCount;
    
    ObjectData(){}
    
    ObjectData(EntryData *e, IndexSize i, SubIndexSize c)
    : entries(e), index(i), entryCount(c)
    {}
};

} /* namespace ppCANOpen */

#endif // PPCAN_OBJECT_DICTIONARY_H
