

#ifndef CANPIPE_H
#define CANPIPE_H

#include "mbed.h"
#include "CircularBuffer.h"

/** Function pointer type for user to attach and handle CAN messages
 */
typedef Callback<int(CANMessage&)> CanMessageCallback;

/** A stack on top of mbed CAN class to handle complex filtered callbacks
 */
class CanPipe
{
public:
    int temp_counter;
    /* Public Type definitions and forward declarations ***********************/
    /** Enumeration for method return values */
    enum CanPipeResult {
        kOkay = 0,
        kErrorCbNodeMemory,
        kErrorCbListMemory,
        kDone
    };

    /** Enumeration for how filters should be handled */
    enum FilterMode {
        kFilterAuto = 0,
        kFilterHardwareOnly,
        kFilterSoftwareOnly
    };

    /* Public Methods *********************************************************/
    /** Creates message handler linked with the CAN device
     *
     * @param p_can reference to CAN device
     * @param filter_mode hardware or software filtering only (Optional)
     *
     * Example:
     * @code
     * #include "mbed.h"
     * #include "CanPipe.h"
     *
     * Ticker ticker;
     * DigitalOut led1(LED1);
     *
     * CAN m_can(P0_11, P0_31);
     * CanPipe m_can_pipe(&m_can);
     *
     * char counter = 0;
     *
     * void send() {
     *     led1 = !led1;
     *     m_can_pipe.PostMessage(CANMessage(1337, &counter, 1));
     * }
     *
     * int callback1(CANMessage &msg) { return CanPipe::kOkay; }
     * int callback2(CANMessage &msg) { return CanPipe::kOkay; }
     * int callback3(CANMessage &msg) { return CanPipe::kOkay; }
     *
     * int main() {
     *     int handle;
     *
     *     handle = m_can_pipe.RegisterFilter(0x200, 0x780);
     *     m_can_pipe.RegisterCallback(callback1, handle);
     *
     *     handle = m_can_pipe.RegisterFilter(0x281, 0);
     *     m_can_pipe.RegisterCallback(callback2, handle);
     *     m_can_pipe.RegisterCallback(callback3, handle);
     *
     *     ticker.attach(send, 1);
     *
     *     while (1) {
     *         __WFI(); //sleep();
     *         m_can_pipe.HandleMessages();
     *     }
     * }
     * @endcode
     */
    CanPipe(CAN *p_can, FilterMode filter_mode = kFilterAuto);

    /** Assigns a filter to apply to CAN messages.
     * @param id 29 bit identifier to base filter on
     * @param mask Bit mask applied to the id
     * @param format CAN message format (Default CANAny)
     * @param handle Number to associate with message when passing this filter (Optional)
     *
     * Can create software filters if device fails to create filters in hardware
     */
    int RegisterFilter(unsigned int id, unsigned int mask, CANFormat format = CANAny, int handle = 0);

    /** Assigns a callback to apply to CAN messages associated with a given filter.
     * @param callback 29 bit identifier to base filter on
     * @param handle Filter handle to associate with this callback (Optional)
     *
     * Member functions can be added using the right callback Constructor
     * @code
     * canopen_class.RegisterCallback(
     *         CanMessageCallback(&object_instance, &ObjectClass::HandleMessage),
     *         handle);
     * @endcode
     */
    int RegisterCallback(CanMessageCallback callback, int handle);

    /** Stage a message to be written to the bus.
     *
     * @param msg message to write
     */
    void PostMessage(CANMessage msg);

    /** Passes all received messages through the software-filters, if any, and
     *  writes any messages that have been posted.
     *  @return Boolean 1 if a message was handled (post or receive).  0 otherwise.
     */
    int HandleMessages();

private:
    /* Private Type definitions and forward declarations **********************/
    static const int kMaxHandles = 32;
    static const int kMaxCallbacks = 64;

    struct SoftwareFilter {
        unsigned int id;
        unsigned int mask;
        CANFormat format;
        int handle;
    };

    struct CallbackNode {
        CanMessageCallback callback;
        CallbackNode *next_node;
    };

    struct CallbackList {
        int handle;
        CallbackNode *begin;
    };

    /* Private Members ********************************************************/
    CAN *p_can_;

    FilterMode filter_mode_;

    SoftwareFilter software_filters_[kMaxHandles];
    CallbackNode callback_node_pool_[kMaxCallbacks];
    CallbackList callback_list_map_[kMaxHandles];

    int num_software_filters_;
    int num_nodes_;
    int num_lists_;

    CircularBuffer<CANMessage, 16> read_buffer;
    CircularBuffer<CANMessage, 16> write_buffer;

    /* Private Methods ********************************************************/
    void ReadIrq(void);

    int FilterMessage();

    CallbackNode* AllocateNode(CanMessageCallback callback);
    CallbackList* AllocateList(int handle);

    CallbackList* FindListWithHandle(int handle);
};

#endif /* CANPIPE_H */
