#include "CanPipe.h"

/* Public Methods ************************************************************/
CanPipe::CanPipe(CAN *p_can, FilterMode filter_mode):
        p_can_(p_can),
        filter_mode_(filter_mode)
{
    num_software_filters_ = 0;
    p_can_->attach<CanPipe>(this, &CanPipe::ReadIrq);
}

int CanPipe::RegisterFilter(unsigned int id,
        unsigned int mask,
        CANFormat format,
        int handle)
{
    printf("CanPipe::RegisterCallback:\r\n");
    int retval = 0;

    switch (filter_mode_) {
        case kFilterHardwareOnly:
            //printf("creating hardware filter\r\n");
            retval = p_can_->filter(id, mask, format, handle);
            break;
        case kFilterAuto:
            //printf("creating hardware filter\r\n");
            retval = p_can_->filter(id, mask, format, handle);
            handle = retval;
        case kFilterSoftwareOnly:
            //printf("   creating software filter\r\n");
            if (num_software_filters_ < kMaxHandles) {
                if (!handle) {
                    handle = 1;
                    for (int i_filter = 0; i_filter < num_software_filters_; ++i_filter) {
                        if (handle <= software_filters_[i_filter].handle) {
                            handle = software_filters_[i_filter].handle + 1;
                        }
                    }
                }
                SoftwareFilter new_filter = {id, mask, format, handle};
                software_filters_[num_software_filters_++] = new_filter;
                retval = handle;
            }
            break;
        default:
            break;
    }
    //printf("\r\n");

    return retval;
}

int CanPipe::RegisterCallback(CanMessageCallback callback, int handle) {
    printf("CanPipe::RegisterCallback:\r\n   handle: %d\r\n", handle);
    /* get new node from the pool */
    CallbackNode *new_node = AllocateNode(callback);
    if (!new_node)
        return kErrorCbNodeMemory;

    /* get the appropriate list */
    CallbackList *list = FindListWithHandle(handle);
    if (!list) {
        /* no list for handle yet.  Allocate it now */
        list = AllocateList(handle);
    }
    if (!list)
        return kErrorCbListMemory;

    /* add new node to list */
    if (list->begin) {
        /* append to the end */
        CallbackNode *node_iterator = list->begin;
        while (node_iterator->next_node) {
            node_iterator = node_iterator->next_node;
        }
        node_iterator->next_node = new_node;
    } else {
        /* this is the first node */
        list->begin = new_node;
    }
    return kOkay;
}


void CanPipe::PostMessage(CANMessage msg) {
    write_buffer.push(msg);
}

int CanPipe::HandleMessages(void) {
    int b_retval = 0; // boolean for whether or not a message was handled

    /* Perform callbacks for all received messages */
    CANMessage msg;
    while (read_buffer.pop(msg)) {
        b_retval = 1;

        printf("* Reading message 0x%03X\r\n", msg.id);
        int handle = 0;

        // software filters override any hardware filtering
        if (filter_mode_ == kFilterAuto || filter_mode_ == kFilterSoftwareOnly) {
            SoftwareFilter filter;
            for (int f = 0; f < num_software_filters_; ++f) {
                filter = software_filters_[f];
                if (filter.format == CANAny || msg.format == filter.format) {
                    if ((msg.id & filter.mask) == (filter.id & filter.mask)) {
                        handle = filter.handle;
                        break;
                    }
                }
            }
        }

        CallbackList *callback_list = FindListWithHandle(handle);
        if (!callback_list)
            return 0; 

        CallbackNode *node_iterator = callback_list->begin;
        while (node_iterator) {
            int result = node_iterator->callback(msg);

            /* if an error occurred or callback claims to have completed message
             * handling, then leave the message chain.
             */
            if (result) {
                break;
            } else {
                node_iterator = node_iterator->next_node;
            }
        }
    }

    /* Write all posted messages to the bus */
    CANMessage write_msg;
    while (write_buffer.pop(write_msg)) {
        b_retval = 1;
        if (p_can_->write(write_msg)) {
            /* Do something */
        } else {
            /* Do something */
        }
    }

    return b_retval;
}

/* Private Methods ************************************************************/
void CanPipe::ReadIrq(void) {

// LPC15xx specific ============================================================
    uint32_t can_int = LPC_C_CAN0->CANINT;
    uint32_t can_status = LPC_C_CAN0->CANSTAT; /* Reading clears the interrupt */
    if (can_int & 0x8000) {
        // TODO: React to status changes
    }
// END LPC15xx specific ========================================================

    CANMessage msg;
    if (p_can_->read(msg)){
        // TODO:  raise flag if buffer is full
        read_buffer.push(msg);
    }
}

CanPipe::CallbackNode* CanPipe::AllocateNode(CanMessageCallback callback) {
    CallbackNode* result = 0;
    if (num_nodes_ < kMaxCallbacks) {
        result = &callback_node_pool_[num_nodes_++];
        result->callback = callback;
    }
    return result;
}

CanPipe::CallbackList* CanPipe::AllocateList(int handle) {
    CallbackList* result = 0;
    if (num_lists_ < kMaxHandles) {
        result = &callback_list_map_[num_lists_++];
        result->handle = handle;
    }
    return result;
}

CanPipe::CallbackList* CanPipe::FindListWithHandle(int handle) {
    CallbackList* result = 0;
    /* iterate over all of the lists */
    for (int i = 0; i < num_lists_; ++i) {
        if (callback_list_map_[i].handle == handle) {
            result = &callback_list_map_[i];
            break;
        }
    }
    return result;
}
