// Unless otherwise noted in individual libraries, this code is for educational
// purposes only with no guaranties - use at your own risk. This file is 
// declared to be completely in the public domain.

#include "mbed.h"
#include "max32630fthr.h"
#include "USBSerial.h"

// for stats
#include "platform/mbed_assert.h"
#include "platform/mbed_debug.h"
#include "platform/mbed_error.h"

// define all stats enabled before including the header
//#define MBED_ALL_STATS_ENABLED 1
// *** using mos 5.11.5 (as 12.0 does not compile even blinky) does not work as expected. must include mbed_app.json file at same
// *** directory level as main.cpp with:
/*
{
    "target_overrides": {
        "*": {
            "platform.all-stats-enabled": true
        }
    }
}
*/
#include "platform/mbed_stats.h"

#define MAX_THREAD_INFO 10

// global vars
mbed_stats_sys_t sys_info;
mbed_stats_cpu_t cpu_info;
mbed_stats_heap_t heap_info;
mbed_stats_stack_t stack_info[ MAX_THREAD_INFO ];

// platform specific
MAX32630FTHR pegasus(MAX32630FTHR::VIO_3V3);

// Virtual serial port over USB
USBSerial microUSB; 

// RTOS is enabled
int main()
{
    // wait until user ready
    while (!microUSB.terminal_connected){
        Thread::yield();
    }


// stats
    microUSB.printf("\n\nSystem:");
    microUSB.printf("\n---");
    mbed_stats_sys_get(&sys_info);
    microUSB.printf("\n\tmbed os version: %d", sys_info.os_version);
    microUSB.printf("\n\tCPU ID: %d", sys_info.cpu_id);
    microUSB.printf("\n\tCompiler ID: ");
    switch(sys_info.compiler_id){
        case ARM:       microUSB.printf("ARM"); break;
        case GCC_ARM:   microUSB.printf("GCC_ARM"); break;
        case IAR:       microUSB.printf("IAR"); break;
        default:        microUSB.printf("UNKNOWN: %d", sys_info.compiler_id); break;
    }
    microUSB.printf("\n\tCompiler version: %d", sys_info.compiler_version);
    
    // times are sourced in uSec, converted to seconds
    microUSB.printf("\n\nCPU (seconds):");
    microUSB.printf("\n---");
    mbed_stats_cpu_get(&cpu_info);
    microUSB.printf("\n\tUp time sec: %0.6f", cpu_info.uptime / 1000000.0);
    microUSB.printf("\n\tIdle time: %0.6f", cpu_info.idle_time / 1000000.0);
    microUSB.printf("\n\tSleep time: %0.6f", cpu_info.sleep_time / 1000000.0);
    microUSB.printf("\n\tDeep sleep time: %0.6f", cpu_info.deep_sleep_time / 1000000.0);
    
    microUSB.printf("\n\nMemory:");
    microUSB.printf("\n---");
    mbed_stats_heap_get( &heap_info );
    microUSB.printf("\n\tBytes allocated currently: %d", heap_info.current_size);
    microUSB.printf("\n\tMax bytes allocated at a given time: %d", heap_info.max_size);
    microUSB.printf("\n\tCumulative sum of bytes ever allocated: %d", heap_info.total_size);
    microUSB.printf("\n\tCurrent number of bytes allocated for the heap: %d", heap_info.reserved_size);
    microUSB.printf("\n\tCurrent number of allocations: %d", heap_info.alloc_cnt);
    microUSB.printf("\n\tNumber of failed allocations: %d", heap_info.alloc_fail_cnt);
    
    mbed_stats_stack_get( &stack_info[0] );
    microUSB.printf("\n\nCumulative Stack:");
    microUSB.printf("\n---");
    microUSB.printf("\n\tMaximum number of bytes used on the stack: %d", stack_info[0].max_size);
    microUSB.printf("\n\tCurrent number of bytes allocated for the stack: %d", stack_info[0].reserved_size);
    microUSB.printf("\n\tNumber of stacks stats accumulated in the structure: %d", stack_info[0].stack_cnt);
    
    mbed_stats_stack_get_each( stack_info, MAX_THREAD_INFO );
    microUSB.printf("\n\nThread Stack:");
    microUSB.printf("\n---");
    for(int i=0;i < MAX_THREAD_INFO; i++) {
        if(stack_info[i].thread_id != 0) {
            microUSB.printf("\n\tThread: %d", i);
            microUSB.printf("\n\t\tThread Id: 0x%08X", stack_info[i].thread_id);
            microUSB.printf("\n\t\tMaximum number of bytes used on the stack: %d", stack_info[i].max_size);
            microUSB.printf("\n\t\tCurrent number of bytes allocated for the stack: %d", stack_info[i].reserved_size);
            microUSB.printf("\n\t\tNumber of stacks stats accumulated in the structure: %d", stack_info[i].stack_cnt); 
            microUSB.printf("\n---");
        }        
    }
    
    microUSB.printf("\n===\n\n");

}

