#include <time.h>
#include "mbed.h"
#include "HoneypotServer.h"
//#define DEBUG
#include "hl_debug.h"

/* Constructor */
/* initialize all members and set the standard error handler. */
HoneypotServer::HoneypotServer()
    : m_pEthernet(NULL)
{
}

HoneypotServer::~HoneypotServer()
{
    if (m_pEthernet == NULL) {
        INFO("Deleting EthernetInterface Object.\n");
        delete m_pEthernet;
    }
}


bool HoneypotServer::start(int * ports, int count, char * master_addr , EthernetInterface* pEthernet)
{
    portsCount = 0;
    //  If no ethernet interface is provided, instantiate own on the heap. This has to be deleted later in the destructor.
    //  If a valid pointer to an thernet interface is proveded, we can simply use it.
    if (pEthernet == NULL) {
        ERR("Socket is not opened!!\r\n");
        return false;

    } else {
        //  In the case that the ethernet interface is provided, it is assumed that a connection has already been created.
        INFO("Using connection IP %s", pEthernet->getIPAddress());
    }

    if(count < 1 && count > MAX_SOCKETS) {
        ERR("Sockets Limited %d \r\n", MAX_SOCKETS);
        return false;
    }


    SetTime();


    portsCount = count;
    portsTriggers = ports;
    masterIP = master_addr;

    for(int i = 0; i < portsCount; i++) {
        if(!Binding(m_Svr[i], ports[i])) {
            ERR("Unable Binding Port: %i \r\n", ports[i]);
        } else {
            INFO("Binding Complete: %i \r\n", ports[i]);
        }
    }

    return true;
}

bool HoneypotServer::Binding(TCPSocketServer& svr, int port)
{
    if (svr.bind(port) < 0) {
        return false;
    }

    if (svr.listen(1) < 0) {
        return false;
    };

    //  set into non blocking operation
    svr.set_blocking(false, 200);

    return true;
}

int HoneypotServer::poll(bool blocking)
{
    for(int i = 0; i < portsCount; i++) {
        TCPSocketConnection connection;

        int port = m_Svr[i].getPort();
        int code = m_Svr[i].accept(connection);

        int isMaster = 0;
        if(strcmp(connection.get_address(), masterIP) == 0) {
            isMaster = 1;
        }

        char data [MAX_LOG_SIZE];
        char file [MAX_LOG_FILE];

        time_t seconds = time(NULL);

        if(code == WIZnet_Chip::SOCK_CLOSED || code == WIZnet_Chip::SOCK_CLOSE_WAIT) {

            sprintf (data, "Client (IP=%s) is reset \r\nPort: %i \r\nDate: %s \r\n\r\n", connection.get_address(), port, ctime(&seconds));
            WriteLog(data, file);
            
            Binding(m_Svr[i], port);

        } else {
            if(code == WIZnet_Chip::SOCK_ESTABLISHED) {

                sprintf (data, "Client (IP=%s) is connected\r\nPort: %i\r\nDate: %s \r\n\r\n", connection.get_address(), port, ctime(&seconds));
                WriteLog(data, file);
                
                if(isMaster == 0) {
                    hits[i] = hits[i] + 1;
                }
                ProcessPoll(connection, port, isMaster);

            }

        }
    }

    return 0;
}

int HoneypotServer::ProcessPoll(TCPSocketConnection CInt, int port, int isMaster)
{


    //handle http requests
    if(port == HTTP_PORT) {
        HTTPConnection con(CInt);

        int c = con.poll();
        if (c == 0) {
            //  Handle the request
            INFO("Handling request !\r\n");
            HandleRequest(con.m_Msg, CInt, isMaster);
        }

    } else {
        //unprocessing requests
        CInt.close();
    }

    INFO("Leaving polling thread\r\n");
    return 0;

}

void HoneypotServer::HandleRequest(HTTPConnection::HTTPMessage& msg, TCPSocketConnection& tcp, int isMaster)
{
    std::string localPath;
    std::map<std::string, HTTPRequestHandler*(*)(const char*, const char*, HTTPConnection::HTTPMessage&, TCPSocketConnection&, int, int*, int, int*), handlersComp>::const_iterator it;

    //  Iterate through registered handlers and check if the handler's path is a subset of the requested uri.
    for (it = m_lpHandlers.begin() ; it != m_lpHandlers.end() ; it++) {
        //  check if this entries' path is fully contained at the beginning of the requested path
        std::string curpth = it->first;

        if (msg.uri.find(curpth) == 0) {
            // firts matching handler found, we just take it and we'll be happy
            localPath = msg.uri.substr(curpth.length());
            break;
        }
    }

    if (it == m_lpHandlers.end()) {
        //  There is no such handler, so return invalid

        INFO("Webrequest left unhandled.\r\n");
    } else {
        //  Valid handler was found
        INFO("Routing webrequest !\r\n");
        //  Instantiate the handler object (handling will be done from withing the object's constructor

        HTTPRequestHandler *phdl = (*it->second)(it->first.c_str(), localPath.c_str(), msg, tcp, isMaster, portsTriggers, portsCount, hits);
        //  now we can delete the object, because handling is completed.
        if (phdl != NULL)
            delete phdl;
    }
}

int HoneypotServer::SetTime()
{
    NTPClient ntp;

    if (ntp.setTime("0.pool.ntp.org") == 0) {
        return 0;
    }
    printf("Error\r\n");

    return -1;
}

int HoneypotServer::WriteLog(char * data, char * fileName)
{
    std::string path = "/local/";
    path.append(fileName);

    FILE * pFile;
    pFile = fopen(path.c_str(),"a");
    if (pFile!=NULL) {
        fputs (data, pFile);
        fclose (pFile);
    }

    return 0;
}

