#ifndef __HONEYPOTSERVER_H__
#define __HONEYPOTSERVER_H__
#include "mbed.h"
#include "EthernetInterface.h"
#include "HTTPConnection.h"
#include "HTTPRequestHandler.h"
#include "NTPClient.h"

#include <map>
using std::map;
#include <string>
using std::string;

#include <TCPSocketConnection.h>
#include <TCPSocketServer.h>

typedef void (*HTTPRequestHandlerFunction)(HTTPConnection::HTTPMessage&, TCPSocketConnection&, int, int*, int, int*);

class HoneypotServer
{
    static const int MAX_SOCKETS = 7;
    
    static const int MAX_LOG_SIZE = 100;
    static const int MAX_LOG_FILE = 15;
    
    
    static const int HTTP_PORT = 80;
    //define more here! 

    TCPSocketServer         m_Svr[MAX_SOCKETS];
    int                     hits[MAX_SOCKETS];
    char*                   masterIP;
    
    bool                    m_bServerListening;
    EthernetInterface*      m_pEthernet;

public:

    /** Constructor for HTTPServer objects.
    */
    HoneypotServer();

    /** Destructor for HTTPServer objects.
    */
    ~HoneypotServer();

    /**
    * Structure which will allow to order the stored handlers according to their associated path.
    */
    struct handlersComp { //Used to order handlers in the right way
        bool operator() (const string& handler1, const string& handler2) const {
            //The first handler is longer than the second one
            if (handler1.length() > handler2.length())
                return true; //Returns true if handler1 is to appear before handler2
            else if (handler1.length() < handler2.length())
                return false;
            else //To avoid the == case, sort now by address
                return ((&handler1)>(&handler2));
        }
    };

    /**
    * Adds a request handler to the handlers list. You will have to use one of the existing implementations.
    * With each handler a \c uri or \c path is associated. Whenever a request is received the server will
    * walk through all registered handlers and check which \c path is matching.
    * @param T : class which will be instanciated to serve these requests for the associated \b path.
    * @param path : request uri starting with this \c path will be served using this handler.
    */
    template<typename T>
    void addHandler(const char* path) {
        m_lpHandlers[path] = &T::create;
    }


    /** Binds server to a specific port and starts listening. This member prepares the internal variables and the server socket
    * and terminates after successfull initialization
    * @param port : port on which to listen for incoming connections
    * @param pEthernet : a pointer to an existing EthernetInterface object or NULL if the HTTPServer shall allocate the object. _Please note that for compatibility reasons
    * your should consider to create the EthernetInterface as a static variable. Otherwise the the object will be created on the heap._
    * @returns : false if an unrecoverable error occured or if the ethernet interface was not set or not initialized correctly, or true if everything was ok.
    */
    bool start(int* ports, int count, char * master_addr, EthernetInterface* pEthernet = NULL);

    /** Performs the regular polling of the server component. Needs to be called cyclically.
    * The function will internally check whether new connections are requested by a client and will also poll all existing client connections.
    * @param blocking : if true,
    * @returns -1 if there was a problem. If 0 is returned, the latest request was served successfully and the server is
    * ready for processing the next request. Simply call \c poll as long as you want to serve new incoming requests.
    */
    int poll(bool blocking = true);

private:

    int portsCount;
    int * portsTriggers;
    

    /** Internal function which processes a request and which will try to find the matching handler function
    * for the given request. Please note that the function will search through the list of handlers, iterating
    * from longest to shortest \c paths. If the registered \c path is a subset of the request the associated
    * handler is considered as being a match.
    * @param msg : Request message data. Contains the requested logical \c uri.
    * @param tcp : Socket to be used for communication with the client.
    */
    void HandleRequest(HTTPConnection::HTTPMessage& msg, TCPSocketConnection& tcp , int isMaster);

    /** Map of handler objects. Can be any object derived from \ref HTTPRequestHeader. Use the \ref addHandler function
    * to register new handler objects.
    */
    map<string, HTTPRequestHandler* (*)(const char*, const char*, HTTPConnection::HTTPMessage&, TCPSocketConnection&, int, int*, int, int*), handlersComp>   m_lpHandlers;

    HTTPRequestHandlerFunction m_pErrorHandler;

    int ProcessPoll(TCPSocketConnection CInt, int port, int isMaster);

    bool Binding(TCPSocketServer& svr, int port);

    int SetTime();
    
    int WriteLog(char * data, char * fileName);
};

#endif //