#ifndef Thermistor_H
#define Thermistor_H

#include "mbed.h"
#include <math.h>
/**
* Reads the resistance of thermistor and using Steinhart–Hart equation 
* converts it to Celsius
* 
*/
class Thermistor{  
public:
    /**
    * Constructor. In the calculations used  a model of 
    * the resistance of a semiconductor at different temperatures - 
    * The Steinhart–Hart equation. Need to set the coefficients a, b, c.
    *
    * @param inputChanel The analog input is connected to the sensor
    * @param a Steinhart–Hart coefficient
    * @param b Steinhart–Hart coefficient
    * @param c Steinhart–Hart coefficient
    */
    Thermistor(AnalogIn inputChanel, double a, double b, double c);
    
    /**
    * The temperature in degrees Celsius
    * @returns Temperature
    */
    double getTemperature();
    
    /**
    * The temperature in degrees Celsius, calculated based on a parameter.
    * Сomfortably for testing.
    *
    * @param adcVal ADC value on the port to which the sensor is connected
    * @param a Steinhart–Hart coefficient
    * @param b Steinhart–Hart coefficient
    * @param c Steinhart–Hart coefficient
    * @param error Error in Ohms (0 by default)
    *
    * @returns Temperature
    */
    static double getTemperatureByAdcValue(float adcVal, double a, double b, double c, double error = 0);
    
    /**
    * Set error value used in the calculation. 
    * The resulting resistance is computed as <real resistance> - error;
    * @param error Error in Ohms
    */    
    void setError(double error);
    
    /**
    * Error value used in the calculation
    * The resulting resistance is computed as <real resistance> - error;
    * @returns Error in Ohms
    */
    double getError();
    
    /**
    * Set a coefficient of Steinhart–Hart equation
    * @param a a-coefficient 
    */
    void setCoefficientA(double a);

    /**
    * Set b coefficient of Steinhart–Hart equation
    * @param b b-coefficient 
    */    
    void setCoefficientB(double b);

    /**
    * Set c coefficient of Steinhart–Hart equation
    * @param c c-coefficient 
    */    
    void setCoefficientC(double c);
    
    /**
    * a coefficient of Steinhart–Hart equation
    * @returns a coefficient 
    */
    double getCoefficientA();
    
    /**
    * b coefficient of Steinhart–Hart equation
    * @returns b coefficient 
    */
    double getCoefficientB();
    
    /**
    * c coefficient of Steinhart–Hart equation
    * @returns c coefficient 
    */    
    double getCoefficientC();
    
private:
    AnalogIn input;
    //Thermistor error in Ohms
    double error;
    double a, b, c;
};

/**
* Tested the class Thermistor
*/
class ThermistorTest{
public:
    static bool adcValue_0_049_is_0_degree();
    static bool adcValue_0_104_is_15_degree();  
    static bool adcValue_0_202_is_30_degree();    
};

#endif