/**
  Copyright (c) 2020 https://www.esmacat.com/

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.

  EsmacatShield.h - Library for using EtherCAT Arduino Shield by Esmacat(EASE).
  Created by Esmacat, 01/22/2020
  
*******************************************************************************
* @file EsmacatShield.h
*******************************************************************************
*/


#ifndef EsmacatShield_h
#define EsmacatShield_h

#include "mbed.h"

#define EASE_READ_REG        0B00000000
#define EASE_WRITE_REG       0B10000000
#define EASE_SINGLE_SHOT     0B10111111
#define EASE_LED_ON          0B00000100
#define EASE_LED_OFF         0B11111011


/**
* @brief This is a driver code for EASE with form factor of Arduino 
*        UNO shields. This shield communicates with the base board via SPI.
*
* @version 1.0
*
* @details EASE stacks onto Arduino-like boards (a.k.a Base board). 
*  The Base board can read and write registers on EASE via SPI.
*  The same registers can also be written and read by an EtherCAT master via a 
*  standard EtherCAT protocol. Multiple EASE boards can be connected with 
*  Ethernet cables with a daisy-chain topology and powered via Ethernet cables 
*  with Power-over-EtherCAT (POE) technology. EASE, thus bridges the data 
*  packets between an EtherCAT master and multiple Base boards.
*        
* @code
*#include "mbed.h"
*#include <EsmacatShield.h>      //Include Esmacat Arduino Library
*
*int counter;
*int v[8];                       //EASE 8 registers
*Serial pc(USBTX, USBRX);
*
*DigitalOut selectPin(D10); // D10 is used to drive chip enable low
*SPI spi(D11, D12, D13); // mosi, miso, sclk
*
*
*int main() 
*{
*
*    EsmacatShield slave(spi, selectPin);        //Define Chip Selector Pin
*    
*    slave.setup_spi();      //Setup SPI for EASE, 8 bit Data, Mode 1, 3MHz freq
*    
*    while(1)
*    {
*      slave.get_ecat_registers(v);  //read all registers
*      slave.write_reg_value(0,counter++, true);   //Write register data (register,value, led_on)
*      wait_us(1000000);                      //sleep for 1000 ms
*      slave.write_reg_value(0,counter++, false);   //Write register data (register,value, led_on)
*      wait_us(1000000);                        //sleep for 1000 ms
*      pc.printf("Next Iteration \n");
*      for (int i=0;i<8;i++)     //print the registers read on pc
*      {
*        pc.printf("%d",i);
*        pc.printf("\t");
*        pc.printf("%d",v[i]);
*        pc.printf("\n");
*      }  
*    }
*
*}
* @endcode
*/


class EsmacatShield
{
public:
    /**********************************************************//**
    * @brief Constructor for EsmacatShield Class.  
    * 
    * @details Requires an existing SPI object as well as a DigitalOut object. 
    * The DigitalOut object is used for a chip enable signal
    *
    * On Entry:
    *     @param[in] spi - pointer to existing SPI object
    *     @param[in] pin - pointer to a DigitalOut pin object
    *
    * On Exit:
    *
    * @return None
    **************************************************************/
  EsmacatShield(SPI &spi, DigitalOut &pin);
    /**********************************************************//**
    * @brief Set up spi communication.  
    * 
    * @details Sets up SPI communication by using the member. 
    * functions format and frequency. 8 bit data, Mode 1,
    * frequency of communication set to 3000000Hz
    *
    * On Entry:
    *     
    * On Exit:
    *
    * @return None
    **************************************************************/
  void setup_spi();
     /**********************************************************//**
    * @brief Write given Register with the given value.  
    * 
    * @details Using this function 8 different 16 bit integers can be 
    *          written into EASE.
    *
    * On Entry:
    *     @param[in] write_addr - The Register number to be written into. 
    *                Can take values from 0 to 7
    *     @param[in] value - The value to be written into the register
    *     @param[in] led_on - True to turn LED ON
    *                       - False to turn LED OFF
    *
    * On Exit:
    *
    * @return None
    **************************************************************/ 
  void write_reg_value(int write_addr,int16_t value, bool led_on=1);
  
     /**********************************************************//**
    * @brief Read EASE Register.  
    * 
    * @details Using this function 8 different 16 bit integers on EASE can be
    *          read.
    *
    * On Entry:
    *     @param[in] Pointer to integer - Eight consecutive memory locations
                     from this pointer will be stored with the integer values read
    *                from EASE.
    *
    * On Exit:
    *     @param[out] pointer to the integer where the data is stored.
    
    *
    * @return None
    **************************************************************/
  int16_t* get_ecat_registers(int16_t regs[8]); 

    /************************************************************
    * @brief Default destructor for EsmacatShield Class.  
    *
    * @details Destroys SPI object if owner 
    *
    * On Entry:
    *
    * On Exit:
    *
    * @return None
    **************************************************************/
 
  ~EsmacatShield();
private:
  
  SPI &ecat_spi;
  DigitalOut &ecat_cs;
  int16_t read_reg_value(int16_t read_addr);
};


#endif

