/** A oled i2c class partial porting from Arduino's OLED_I2C library
 *
 *  @author  Poushen Ou
 *  @version 1.0
 *  @date    17-Jun-2018
 *
 *  This code provide operation for I2C OLED SSD1306
 *
 *  About OLED SSD1306
 *      https://cdn-shop.adafruit.com/datasheets/SSD1306.pdf
 */
/*
  OLED_I2C.h - Arduino/chipKit library support for 128x64 pixel SSD1306 OLEDs
  Copyright (C)2015 Rinky-Dink Electronics, Henning Karlsen. All right reserved

  This library has been made to make it easy to use 128x64 pixel OLED displays
  based on the SSD1306 controller chip with an Arduino or a chipKit.

  You can always find the latest version of the library at
  http://www.RinkyDinkElectronics.com/

  This library is free software; you can redistribute it and/or
  modify it under the terms of the CC BY-NC-SA 3.0 license.
  Please see the included documents for further information.

  Commercial use of this library requires you to buy a license that
  will allow commercial use. This includes using the library,
  modified or not, as a tool to sell products.

  The license applies to all part of the library including the
  examples and tools supplied with the library.
*/

#ifndef _OLED_I2C_H_
#define _OLED_I2C_H_

#include "mbed.h"
#include "eeprom.h"

#define LEFT    0
#define RIGHT   9999
#define CENTER  9998

#define SSD1306_128_64

#define SSD1306_ADDR        0x78
#define SSD1306_MEM_SIZE    360
#define SSD1306_WIDTH       72
#define SSD1306_LENGTH      40

#define SSD1306_COMMAND         0x00
#define SSD1306_DATA            0xC0
#define SSD1306_DATA_CONTINUE   0x40

//#define SSD1306_DATA_OUT_OPTIMIZED

// SSD1306 Commandset
// ------------------
// Fundamental Commands
#define SSD1306_SET_CONTRAST_CONTROL        0x81
#define SSD1306_DISPLAY_ALL_ON_RESUME       0xA4
#define SSD1306_DISPLAY_ALL_ON              0xA5
#define SSD1306_NORMAL_DISPLAY              0xA6
#define SSD1306_INVERT_DISPLAY              0xA7
#define SSD1306_DISPLAY_OFF                 0xAE
#define SSD1306_DISPLAY_ON                  0xAF
#define SSD1306_NOP                         0xE3
// Scrolling Commands
#define SSD1306_HORIZONTAL_SCROLL_RIGHT                 0x26
#define SSD1306_HORIZONTAL_SCROLL_LEFT                  0x27
#define SSD1306_HORIZONTAL_SCROLL_VERTICAL_AND_RIGHT    0x29
#define SSD1306_HORIZONTAL_SCROLL_VERTICAL_AND_LEFT     0x2A
#define SSD1306_DEACTIVATE_SCROLL                       0x2E
#define SSD1306_ACTIVATE_SCROLL                         0x2F
#define SSD1306_SET_VERTICAL_SCROLL_AREA                0xA3
// Addressing Setting Commands
#define SSD1306_SET_LOWER_COLUMN                        0x00
#define SSD1306_SET_HIGHER_COLUMN                       0x10
#define SSD1306_MEMORY_ADDR_MODE                        0x20
#define SSD1306_SET_COLUMN_ADDR                         0x21
#define SSD1306_SET_PAGE_ADDR                           0x22
// Hardware Configuration Commands
#define SSD1306_SET_START_LINE                          0x40
#define SSD1306_SET_SEGMENT_REMAP                       0xA0
#define SSD1306_SET_MULTIPLEX_RATIO                     0xA8
#define SSD1306_COM_SCAN_DIR_INC                        0xC0
#define SSD1306_COM_SCAN_DIR_DEC                        0xC8
#define SSD1306_SET_DISPLAY_OFFSET                      0xD3
#define SSD1306_SET_COM_PINS                            0xDA
#define SSD1306_CHARGE_PUMP                             0x8D
// Timing & Driving Scheme Setting Commands
#define SSD1306_SET_DISPLAY_CLOCK_DIV_RATIO             0xD5
#define SSD1306_SET_PRECHARGE_PERIOD                    0xD9
#define SSD1306_SET_VCOM_DESELECT                       0xDB

/** oled_i2c Class Library
 * to provide access interface to OLED screen for mbed
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "oled_i2c.h"
 * #include "oled_font.h"
 *
 * // make oled_i2c instance using I2C object.
 * // with default slave address 0x78 (0x3C in 7bit format)
 * // test ok with SSD1306 I2C 128*64 oled screen
 * I2C i2c(dp5,dp27);   // for LPC1114 or LPC1115
 * oled_i2c oled(i2c);
 *
 * int main()
 * {
 *     i2c.frequency(1000 * 1000);
 *     oled.init_oled();
 * 
 *     oled.setFont(BigNumbers);
 *     //oled.print("27", 20, 10);
 *     //oled.printNumI(27, 0, 0);
 *     oled.printNumF(27.45, 2, 0, 0);
 *     oled.setFont(SmallFont);
 *     oled.print("~c", 60, 30);
 *     oled.update();
 *
 *     while(1);
 * }
 * @endcode
 */
class oled_i2c
{
public:
    /** Create a oled_i2c instance connected to specified I2C pins with specified address
     *
     * @param i2c_obj I2C object (instance)
     * @param address (option) I2C-bus slave address
     */
    oled_i2c(I2C &i2c_obj, char address = SSD1306_ADDR);

    /** Destractor */
    ~oled_i2c();

    /** Initialization */
    void init(void);
    
    /** send init command to SSD1306 */
    void init_oled(void);
    
    /** update screen buffer */
    void update(void);
    
    /** clear screen */
    void clrScr(void);
    
    /** fill screen */
    void fillScr(void);
    
    /** set pixel on coordinate x, y
     *
     * @param x coordinate x (column)
     * @param y coordinate y (row)
     */
    void setPixel(uint16_t x, uint16_t y);
    
    /** clear pixel on coordinate x, y
     *
     * @param x coordinate x (column)
     * @param y coordinate y (row)
     */
    void clrPixel(uint16_t x, uint16_t y);
    
    /** set font type
     *
     * @param font the font to print
     */
    void setFont(const uint8_t* font);
    
    /** print character to coordinate x, y
     *
     * @param c the character to print
     * @param x coordinate x
     * @param y coordinate y
     */
    void print_char(unsigned char c, int x, int y);
    
    /** print string to coordinate x, y
     *
     * @param st the string to print
     * @param x coordinate x
     * @param y coordinate y
     */
    void print(char* st, int x, int y);
    
    /** print integer number to coordinate x, y
     *
     * @param num the integer to be print
     * @param x coordinate x
     * @param y coordinate y
     * @param length the length of integer to be print
     * @param filler the prefix character
     */
    void printNumI(long num, int x, int y, int length=0, char filler=' ');
    
    /** print float number to coordinate x, y
     *
     * @param num the float number to be print
     * @param dec the numbers of digits after point
     * @param x coordinate x
     * @param y coordinate y
     * @param divider should be the point (.)
     * @param length the length of float number to be print
     * @param filler the prefix character
     */
    void printNumF(double num, uint8_t dec, int x, int y, char divider='.', int length=0, char filler=' ');
     

private:
    I2C &i2c;
    char adr;
    eeprom *epm_p;
    eeprom &epm;
    struct _current_font
    {
        const uint8_t* font;
        uint8_t x_size;
        uint8_t y_size;
        uint8_t offset;
        uint8_t numchars;
    } cfont;
    uint8_t scrbuf[SSD1306_MEM_SIZE];
    
    /** send SSD1306 command 
     *
     * @param a the SSD1306 command byte to send out
     */
    void comm_out(uint8_t a);
    
    /** send many SSD1306 commands in one I2C transaction
     *
     * @param pPage pointer to command buffer
     * @param size the command buffer size
     */
    void comm_page_out(uint8_t* pPage, uint8_t size);
    
    /** send SSD1306 data
     *
     * @param a the SSD1306 data byte to send out
     */
    void data_out(uint8_t a);
    
    /** send many SSD1306 data bytes in one I2c transaction
     *
     * @param pPage pointer to data buffer
     * @param size the data buffer size
     */
    void data_page_out(uint8_t* pPage, uint8_t size);
    
    /** make all pixel on or off acording to screen buffer */
    void pixel_on(void);
    
};

#endif
