#include "eeprom.h"

eeprom::eeprom(I2C &i2c_obj, char address)
    : i2c(i2c_obj), adr(address)
{
    init();
}

void eeprom::init(void)
{
    i2c.frequency(400*1000);
}

void eeprom::write_address(int address, int address_size, bool repeated)
{   
    uint8_t buffer[4] = { 0, 0, 0, 0 };
    page_write(address, address_size, buffer, 0);
}

uint8_t eeprom::current_read(void)
{
    uint8_t data;
    i2c.read((int)adr,(char *)&data,1);
    return data;
}

void eeprom::sequential_read(uint8_t *buffer, int buffer_size)
{
    i2c.read((int)adr, (char *)buffer, buffer_size);   
}

void eeprom::random_read(int address, int address_size, uint8_t *buffer, int buffer_size)
{
    write_address(address, address_size, true);
    i2c.read((int)adr, (char *)buffer, buffer_size);
}

void eeprom::byte_write(int address, int address_size, uint8_t data, bool repeated)
{   
    uint8_t buffer[4] = { 0, 0, 0, 0 };
    buffer[3] = data;
    page_write(address, address_size, buffer, 1);
}

void eeprom::page_write(int address, int address_size, uint8_t *buffer, int buffer_size, bool repeated)
{
    // for simplify, buffer must prefix 3 bytes for memory address
    switch (address_size) {
        case ONE_BYTE_ADDRESS:
            buffer[2] = address & 0xFF;
            break;
        case TWO_BYTES_ADDRESS:
            buffer[1] = (address >> 8) & 0xFF;
            buffer[2] = address & 0xFF;
            break;
        case THREE_BYTES_ADDRESS:
            buffer[0] = (address >> 16) & 0xFF;
            buffer[1] = (address >> 8) & 0xFF;
            buffer[2] = address & 0xFF;
            break;
    }
    i2c.write((int)adr,(char *)(buffer + (3 - address_size)),address_size + buffer_size,repeated);
}

void eeprom::ack_polling(void)
{
    uint8_t buffer[] = { 0 };
    int ack;
    
    do {
        ack = i2c.write((int)adr, (char *)buffer, 1, true);
    } while(ack == 1);
    i2c.stop();
}
