/** A class for eeprom accesss operation
 *
 *  @author  Poushen Ou
 *  @version 1.0
 *  @date    15-Jun-2018
 *
 *  This code provide classic access operation for I2C EEPROM
 *
 *  About I2C EEPROM 24FC256:
 *      http://ww1.microchip.com/downloads/en/DeviceDoc/21203M.pdf
 */
#ifndef _EEPROM_H_
#define _EEPROM_H_

#include "mbed.h"

#define EEPROM_ADDR 0xA0

/** eeprom Class Library
 * to provide very simple interface for mbed
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "eeprom.h"
 *
 * #define BUFFER_SIZE  64
 *
 * // make eeprom instance using I2C object.
 * // with default slave address 0xA0 (0x50 in 7bit format)
 * // test ok with 24FC256 EEPROM
 * I2C i2c(dp5,dp27);
 * eeprom epm(i2c);
 *
 * // import!!! must prefix 3 bytes for memory address
 * uint8_t buffer[BUFFER_SIZE + 3]; 
 *
 * int main()
 * {
 *     printf("LPC1114 demo.\n\r");
 *
 *     // -------- page write --------------------
 *     for (int i=0; i<BUFFER_SIZE; i++)
 *         buffer[i+3] = i;     // for prefix 3 memory address
 *
 *     epm.page_write(12288, TWO_BYTES_ADDRESS, buffer, BUFFER_SIZE);
 *     //wait(0.008);
 *     epm.ack_polling();
 *
 *     // -------- current read ------------------
 *     printf("below shold be 00 01 ... 3f\n\r");
 *     epm.write_address(12288, TWO_BYTES_ADDRESS);
 * 
 *     for (int i=0; i<BUFFER_SIZE; i++) {
 *       printf("%.2x ", epm.current_read());
 *     }
 *     printf("\n\r");
 *     
 *     // -------- byte write --------------------
 *     epm.byte_write(12288, TWO_BYTES_ADDRESS, 0xAA);
 *     epm.ack_polling();
 *     
 *     // -------- sequential read ---------------
 *     printf("below shold be aa 01 02 ... 3f\n\r");
 *     epm.write_address(12288, TWO_BYTES_ADDRESS);
 *   
 *     uint8_t data[BUFFER_SIZE];
 *     epm.sequential_read(data, BUFFER_SIZE);
 *     for (int i=0; i<BUFFER_SIZE; i++) {
 *         printf("%.2x ", data[i]);
 *     }
 *     printf("\n\r");
 *
 *     // -------- byte write --------------------
 *     epm.byte_write(12289, TWO_BYTES_ADDRESS, 0xBB);
 *     epm.ack_polling();
 *
 *     // -------- random read -------------------
 *     printf("below shold be aa bb 02 03 ... 3f\n\r");
 *     epm.random_read(12288, TWO_BYTES_ADDRESS, buffer, 64);
 *     for (int i=0; i<64; i++) {
 *         printf("%.2x ", buffer[i]);
 *     }
 *     printf("\n\r");
 *
 *     while(1);
 * }
 * @endcode
 */
class eeprom
{
public:
    /** Create a eeprom instance connected to specified I2C pins with specified address
     *
     * @param i2c_obj I2C object (instance)
     * @param address (option) I2C-bus slave address (default: 0xA0)
     */
    eeprom(I2C &i2c_obj, char address = EEPROM_ADDR);

    /** Initialization */
    void init(void);
    
    /** Write address with specify address size
     *  
     * @param address eeprom memory address
     * @param address_size should be 1 - 3(ONE_BYTE_ADDRESS, TWO_BYTES_ADDRESS, THREE_BYTES_ADDRESS)
     */
    void write_address(int address, int address_size, bool repeated=false);

    /** Current read, read the current memory data
     *
     * @return the current memory data
     */
    uint8_t current_read(void);
    
    /** Sequential read, read one or many bytes from current memory address
     *
     * @param buffer the start address point to buffer
     * @param buffer_size the length of buffer
     */
    void sequential_read(uint8_t *buffer, int buffer_size);
    
    /** Random read, read one or more memory data from assign memory address
     *
     * @param address eeprom memory address
     * @param address_size should be 1 - 3 (ONE_BYTE_ADDRESS, TWO_BYTES_ADDRESS, THREE_BYTES_ADDRESS)
     * @param buffer the start address point to buffer
     * @param buffer_size the length of buffer
     */
     void random_read(int address, int address_size, uint8_t *buffer, int buffer_size);
     
    /** byte write, write one byte to assign memory address
     *
     * @param address eeprom memory address
     * @param address_size should be 1 - 3(ONE_BYTE_ADDRESS, TWO_BYTES_ADDRESS, THREE_BYTES_ADDRESS)
     * @param data the byte data to write
     */
    void byte_write(int address, int address_size, uint8_t data, bool repeated=false);
    
    /** page write, write many bytes to assign memory address
     * , do not deal with page size and aligned problem, be careful to use!!!
     *
     * @param address eeprom memory address
     * @param address_size should be 1 - 3(ONE_BYTE_ADDRESS, TWO_BYTES_ADDRESS, THREE_BYTES_ADDRESS)
     * @param buffer the page data to write
     */
    void page_write(int address, int address_size, uint8_t *buffer, int buffer_size, bool repeated=false);
    
    /** ack polling, wait for EEPROM write completed */
    void ack_polling(void);

private:
    I2C &i2c;
    char adr;
};

enum {
    ONE_BYTE_ADDRESS = 1,
    TWO_BYTES_ADDRESS,
    THREE_BYTES_ADDRESS
};

#endif