/** A practice code for class library development
 *
 *  @author  Yasubumi KANAZAWA
 *  @version 1.0
 *  @date    26-Nov-2017
 *
 *  This code is plactice of the class code development.
 *  The sample is a I2C temperature sensor TMP102.
 *
 *  Reference page
 *      https://os.mbed.com/users/okano/notebook/mbed-library-study-meeting-2014-nov-07/"
 *
 *  About TMP102:
 *      http://www.sparkfun.com/datasheets/Sensors/Temperature/tmp102.pdf
 *
 *
 *  extended by Poushen Ou to provide more API function
 *  17-Jul-2018
 *
 */

#include "mbed.h"

/** Default slave address(ADD0 connected to GND) */
#define ADDR_TMP102     0x90

/** TMP102 register names and addresses */
#define TMP102_Temp     0x00
#define TMP102_Conf     0x01
#define TMP102_Tlow     0x02
#define TMP102_Thigh    0x03

/** test_TMP102 Class Library
 * to provide very simple interface for mbed
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "test_TMP102.h"
 *
 * // make test_TMP102 instance using GPIO pin name.
 * test_TMP102 temp0(dp4,dp5);
 *
 * // make test_TMP102 instance using I2C object.
 * I2C i2c(dp4,dp5);
 * test_TMP102 temp1(i2c);
 *
 * int main()
 * {
 *     float t0,t1;
 *
 *     i2c.frequency(400*1000);
 *
 *     while(1) {
 *         t0=temp0;
 *         t1=temp1;
 *         printf("Temp: %7.3f, %7.3f\r\n", t0, t1);
 *         wait(1.0);
 *     }
 * }
 * @endcode
 */
class test_TMP102
{
public:
    /** Create a test_TMP102 instance connected to specified I2C pins with specified address
     *
     * @param sda I2C-bus SDA pin
     * @param scl I2C-bus SCL pin
     * @param address (option) I2C-bus slave address (default: 0x90)
     */
    test_TMP102(PinName sda, PinName scl, char address = ADDR_TMP102);

    /** Create a test_TMP102 instance connected to specified I2C pins with specified address
     *
     * @param i2c_obj I2C object (instance)
     * @param address (option) I2C-bus slave address (default: 0x90)
     */
    test_TMP102(I2C &i2c_obj, char address = ADDR_TMP102);

    /** Destractor */
    ~test_TMP102();

    /** Initialization */
    void init(void);

    /** Read temperature
     *
     *  @return value of degree Celsius (in float)
     */
    float read(void);
    
    /** Read temperature
     *
     *  @return value of degree C (in float)
     */
    float readTempC(void);
    
    /** Read temperature
     *
     *  @return value of degree F (in float)
     */
    float readTempF(void);

    /** Read temperature
     *
     *  @return the object returns the read value
     */
    operator float(void);
    
    /** Set the conversion rate (0-3) 
     *
     *  @param rate Conversion Rate 
     *          0 - 0.25Hz
     *          1 - 1 Hz
     *          2 - 4 Hz (default)
     *          3 - 8 Hz
     */
    void setConversionRate(char rate);
    
    /** Enable or disable extened mode 
     *
     *  @param mode Extend Mode (0 disable, 1 enabled)
     *
     */
    void setExtendedMode(bool mode);
    
    /** Switch sensor to low power mode */
    void sleep(void);
    
    /** Wakeup and start running in normal power mode */
    void wakeup(void);
    
    /** enter One-Shot mode, and wait for temperature sample complete (about 26ms) */
    void oneShot(void);
    
private:
    I2C *i2c_p;
    I2C &i2c;
    char adr;
};