#ifndef CHANGEME_H_
#define CHANGEME_H_
#include "USBSerial.h"

#include "mbed.h"
#include <string>

#include "FATFileSystem.h"
//#include "SDBlockDevice.h"
#include "SPIFBlockDevice.h"

#include <stdio.h>
#include <errno.h>

//PIN OUT Flash
#define Flash_MOSI PA_7
#define Flash_MISO PA_6
#define Flash_SCK PA_5
#define Flash_CS PB_6

//Pin de l'eeprom I2C
#define SCL_PIN  PB_8
#define SDA_PIN  PB_9

//Adresse i2C de l'eeprom
#define EEPROM_ADDRESS 0xA0

#define DEBUG_MODE 0

#if DEBUG_MODE
#define DEBUG(...) { printf(__VA_ARGS__); fflush(stdout);}
#else
#define DEBUG(...)
#endif

#define DEEP_DEBUG_MODE 1

#if DEEP_DEBUG_MODE
#define DEEP_DEBUG(...) { printf(__VA_ARGS__); fflush(stdout);}
#else
#define DEEP_DEBUG(...)
#endif

/** Utils class.
 *  Rassemblant des fonctions annexes pour le fonctionnement de l'ARNSRS.
 *
 *  A besoin des libs suivantes :
 *
 *  -  FATFileSystem
 *
 *  -  SPIFBlockDevice
 *
 *
 *  Constantes de l'application :
 *
 * Pin out Flash :
 *
 * -  Flash_MOSI PA_7
 *
 * -  Flash_MISO PA_6
 *
 * -  Flash_SCK PA_5
 *
 * -  Flash_CS PB_6
 *
 * Liste des erreurs possibles sur les fonctions Flash :
 *
 *
 * (0) no error
 *
 * (1) A hard error occurred in the low level disk I/O layer
 *
 * (2) Assertion failed
 *
 * (3) The physical drive cannot work
 *
 * (4) Could not find the file
 *
 * (5) Could not find the path
 *
 * (6) The path name format is invalid
 *
 * (7) Access denied due to prohibited access or directory full, Permission denied
 *
 * (8) Access denied due to prohibited access, File exists
 *
 * (9) The file/directory object is invalid, Bad address
 *
 * (10) The physical drive is write protected
 *
 * (11) The logical drive number is invalid
 *
 * (12) The volume has no work area, No such device or address
 *
 * (13) There is no valid FAT volume, No such file or directory
 *
 * (14) The f_mkfs() aborted due to any parameter error
 *
 * (15) Could not get a grant to access the volume within defined period, Bad file number
 *
 * (16) The operation is rejected according to the file sharing policy, Permission denied
 *
 * (17) LFN working buffer could not be allocated, Not enough space
 *
 * (18) Number of open files > _FS_LOCK, Too many open files in system
 *
 * (19) Given parameter is invalid, Exec format error
*/

class UTILS
{

public:

    /**Remapper une valeur dans une autre échelle.
    * @param float Valeur à remapper
    * @param float Minimum de l'échelle de la valeur
    * @param float Maximum de l'échelle de la valeur
    * @param float Minimum de la nouvelle échelle
    * @param float Maximum de de la nouvelle échelle
    */
    static float Remap(float x, float in_min, float in_max, float out_min, float out_max);

    /**Contraint une valeur entre deux limites.
    * @param float Valeur à contraindre
    * @param float Minimum de la valeur
    * @param float Maximum de la valeur
    */
    static float constrain(float x, float a, float b);

    /**Fonction d'affichage des infos de la Flash.
    * @param Serial Port série ou sortir le résultat
    */
    static void Flash_Infos(USBSerial *serial_port = NULL);

    /**Fonction de stockage valeur sur la Flash.
    * @param float Val_To_Store, la valeur à stocker
    * @param char* Nom du fichier ou stocker cette valeur
    */
    static void Store_A_Val(float Val_To_Store, char* File_Name);

    /**Fonction de récupération d'une valeur stockée sur la Flash.
    * @param char* Nom du fichier ou stocker cette valeur
    * @returns
    *   valeur de calibration des capteurs O2
    */
    static float Read_A_Val(char* File_Name);

    /**Fonction d'enregistrement d'une chaine de charatères sur la Flash.
    * @param char* To_Store, chaine de charactères
    * @param char* Nom du fichier, s'il existe il est ouvert, sinon il est créé
    */
    static void Write_Flash_File(char* To_Store, char* File_Name);

    /**Fonction de lecture d'un fichier sur la Flash.
    * @param Serial Port série ou sortir le résultat
    * @param char* Nom du fichier
    */
    static void Read_Flash_File(USBSerial *serial_port, char* File_Name);
    
    /**Fonction de renvoie de la taille d'un fichier.
    * @param Serial Port série ou sortir le résultat
    * @param char* Nom du fichier
    */
    static void Get_File_Size(USBSerial *serial_port, char* File_Name);

    /**Fonction d'éffaçage d'un fichier sur la Flash.
    * @param char* Nom du fichier à effacer
    */
    static void Delete_Flash_File(char* File_Name);

    /**Fonction nettoyage de la Flash.
    */
    static void Clean_Flash();

    /**Fonction nettoyage de la Flash.
    */
    static void Clean_Flash_All();
    
    /**Fonction d'éffaçage d'un fichier sur la Flash.
    * @param char* Nom du fichier à effacer
    * @param char* Nouveau Nom du fichier
    */
    static void Rename_Flash_File(char* Old_File_Name, char* New_File_Name);

    /**Fonction montage de la Flash.
    */
    static void Mount_Flash();

    /**Fonction demontage de la Flash.
    */
    static void UnMount_Flash();

    /**Fonction formatage de la Flash.
    */
    static void Format_Flash();

    /**Fonction DIR.
    * @param Serial Port série ou sortir le résultat
    * @param char* Nom du répertoire
    */
    static void Dir_Flash(USBSerial *serial_port = NULL, char* Dir_Name = "");
    
    /**Fonction de numérotage des fichiers log.
    * @returns
    *   Index du nouveau fichier log
    */
    static int File_Index();

    /**Fonction de test si un fichier existe.
    * @param char* Nom du fichier
    * @returns
    *   true si le fichier existe, false s'il n'existe pas
    */
    static bool File_Exist(char* File_Name);

    /**Fonction d'affichage du menu d'aide.
    * @param Serial Port série ou sortir le résultat
    */
    static void Help(USBSerial *serial_port = NULL);

    /**Fonction d'écriture sur l'eeprom.
    * @param char data à stocker
    * @param int eeadress de stockage de la data dans l'eeprom
    * @param int adress adresse I2C de l'eeprom
    */
    static void write_EEPROM(char *data, unsigned int eeaddress, int address = EEPROM_ADDRESS);

    /**Fonction de lecture d'un int dans l'eeprom.
    * @param int eeadress de stockage de la data dans l'eeprom
    * @param int adress adresse I2C de l'eeprom
    * @returns
    *   valeur int stockée à cette adresse
    */
    static int read_I_EEPROM(unsigned int eeaddress, int address = EEPROM_ADDRESS);
    
    /**Fonction de lecture d'un float dans l'eeprom.
    * @param int eeadress de stockage de la data dans l'eeprom
    * @param int adress adresse I2C de l'eeprom
    * @returns
    *   valeur float stockée à cette adresse
    */
    static float read_F_EEPROM(unsigned int eeaddress, int address = EEPROM_ADDRESS);
    
    /**Fonction de lecture d'un char dans l'eeprom.
    * @param char data ou va être stocké le résultat
    * @param int eeadress de stockage de la data dans l'eeprom
    * @param int adress adresse I2C de l'eeprom
    * @returns
    *   char stockée à cette adresse
    */
    static void read_C_EEPROM(char *data, unsigned int eeaddress , int address = EEPROM_ADDRESS);
    
    /**Fonction de lecture d'un char dans l'eeprom.
    * @param int eeadress de stockage de la data dans l'eeprom
    * @param int adress adresse I2C de l'eeprom
    * @returns
    *   char stockée à cette adresse
    */
    static char* read_EEPROM(unsigned int eeaddress , int address = EEPROM_ADDRESS);
    
    /**Fonction de nettoyage d'une ligne de l'eeprom.
    * @param int eeadress de la ligne a effacer
    * @param int adress adresse I2C de l'eeprom
    */
    static void clean_line_EEPROM(unsigned int eeaddress, int address = EEPROM_ADDRESS);

    /**Fonction de nettoyage l'eeprom.
    * @param int adress adresse I2C de l'eeprom
    */
    static void clean_EEPROM(int address = EEPROM_ADDRESS);
    
    /**Fonction pour mise à jour firmware par USB.
    * @param 
    * @returns
    *   
    */
    static void DFU();
    
private:
//Rien...
};
#endif