/* 
 / _____)             _              | |
( (____  _____ ____ _| |_ _____  ____| |__
 \____ \| ___ |    (_   _) ___ |/ ___)  _ \
 _____) ) ____| | | || |_| ____( (___| | | |
(______/|_____)_|_|_| \__)_____)\____)_| |_|
    (C)2015 Semtech

Description: MBED LoRaWAN example application

License: Revised BSD License, see LICENSE.TXT file include in the project

Maintainer: Miguel Luis and Gregory Cristian
*/
#include <cstdio>
#include <string>
#include <cassert>

#include "mbed.h"
#include "cantcoap.h"

#include "lmic.h"
#include "debug.h"

const std::string REGISTRATION_SEGMENT ="/rd";
const std::string ENDPOINT_SEGMENT     = "?ep=";
const std::string LIFETIME             ="&lt=";
const std::string BINDING              ="&b=";

const std::string REGISTRATION_OPEN  =    "<";
const std::string REGISTRATION_CLOSE =    ">";
const std::string REGISTRATION_SEPARATOR ="/";

int _node_Id=0;

const std::string endPoint_Name = "loraDevice"; 
const int lifeTime = 300;
const std::string binding = "U";

unsigned char * _payload;
long _payload_size;

/*!
 * When set to 1 the application uses the Over-the-Air activation procedure
 * When set to 0 the application uses the Personalization activation procedure
 */
#define OVER_THE_AIR_ACTIVATION                     0

#if( OVER_THE_AIR_ACTIVATION == 0 )

/*!
 * Defines the network ID when using personalization activation procedure
 */
#define LORAWAN_NET_ID                              ( uint32_t )0x00000000

/*!
 * Defines the device address when using personalization activation procedure
 */
#define LORAWAN_DEV_ADDR                            ( uint32_t )0x12345678

#endif

/*!
 * Defines the application data transmission duty cycle
 */
#define APP_TX_DUTYCYCLE                            5000 // 5 [s] value in ms
#define APP_TX_DUTYCYCLE_RND                        1000 // 1 [s] value in ms

/*!
 * LoRaWAN Adaptative Data Rate
 */
#define LORAWAN_ADR_ON                              1

/*!
 * LoRaWAN confirmed messages
 */
#define LORAWAN_CONFIRMED_MSG_ON                    1

/*!
 * LoRaWAN application port
 */
#define LORAWAN_APP_PORT                            15

/*!
 * User application data buffer size
 */
#if ( LORAWAN_CONFIRMED_MSG_ON == 1 )
#define LORAWAN_APP_DATA_SIZE                       6

#else
#define LORAWAN_APP_DATA_SIZE                       1

#endif

#define UINT16_MAX (65535U)
#define UINT64_MAX (18446744073709551615ULL)


std::string to_string( int x ) {
  int length = snprintf( NULL, 0, "%d", x );
  assert( length >= 0 );
  char* buf = new char[length + 1];
  snprintf( buf, length + 1, "%d", x );
  std::string str( buf );
  delete[] buf;
  return str;
}
unsigned char * get_Registration_Payload(long *payload_size){

    string registration_Payload ="";

    string s="";

    s.append(REGISTRATION_OPEN);
    s.append(REGISTRATION_SEPARATOR);
    s.append("3/0/0");
    s.append(REGISTRATION_CLOSE);
    s.append(",");
    s.append(REGISTRATION_OPEN);
    s.append(REGISTRATION_SEPARATOR);
    s.append("3/0/1");
    s.append(REGISTRATION_CLOSE);
    s.append(",");
    s.append(REGISTRATION_OPEN);
    s.append(REGISTRATION_SEPARATOR);
    s.append("3/0/2");
    s.append(REGISTRATION_CLOSE);

    registration_Payload.append(s);
    
    unsigned char *c = new unsigned char[registration_Payload.size()+1];
    copy(registration_Payload.begin(),registration_Payload.end(),c);
    c[registration_Payload.size()]='\0';
    *payload_size=registration_Payload.size();
    
    return c;

}
uint8_t * get_Token(int * size){
    srand(time(0)+_node_Id);
    long test=0;
    bool exist=false;
    
    do{
        test=(rand() % UINT64_MAX);
        
    }while (exist==true);
    uint8_t ones = 0xFF;
    *size=1;
    for (int i=0; i<8; ++i) {
        if ( (test>>8*i & ones) =='\0' || i==8) {
            *size=i;
            break;
        }
    }
    uint8_t * token =new uint8_t[*size];
    for (int i=0; i<*size; ++i){
        token[*size-1-i]=test>>8*i & ones;
    }
    return token;
}

uint16_t get_Message_ID(){
    srand(time(0)+_node_Id);
    int test=0;
    bool exist=false;
    do{
        
        exist=false;
        test=(rand() % UINT16_MAX);
        
    }while (exist==true);
    
    
    return (uint16_t) test;
    
}

char * get_Registration_Query(){
    
    string buffer;
    buffer.append(REGISTRATION_SEGMENT);
    buffer.append(ENDPOINT_SEGMENT);
    buffer.append(endPoint_Name);
    buffer.append(LIFETIME);
    buffer.append(to_string(lifeTime));
    buffer.append(BINDING);
    buffer.append(binding);
    
    char *c = new char[buffer.size()+1];
    copy(buffer.begin(),buffer.end(),c);
    c[buffer.size()]='\0';
    return c;
    

}

//////////////////////////////////////////////////
// CONFIGURATION (FOR APPLICATION CALLBACKS BELOW)
//////////////////////////////////////////////////

// application router ID (LSBF)
static const uint8_t AppEui[8] =
{
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

// unique device ID (LSBF)
static const u1_t DevEui[8] =
{
    0x00, 0x00, 0x00, 0x00, 0x08, 0x06, 0x02, 0x48
//    0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF
//    0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x099, 0xF7
};

// device-specific AES key (derived from device EUI)
static const uint8_t DevKey[16] = 
{
    0x2B, 0x7E, 0x15, 0x16, 0x28, 0xAE, 0xD2, 0xA6,
    0xAB, 0xF7, 0x15, 0x88, 0x09, 0xCF, 0x4F, 0x3C
};

#if( OVER_THE_AIR_ACTIVATION == 0 )
// network session key
static uint8_t NwkSKey[] = 
{ 
    0x2B, 0x7E, 0x15, 0x16, 0x28, 0xAE, 0xD2, 0xA6,
    0xAB, 0xF7, 0x15, 0x88, 0x09, 0xCF, 0x4F, 0x3C
};

// application session key
static uint8_t ArtSKey[] = 
{ 
    0x2B, 0x7E, 0x15, 0x16, 0x28, 0xAE, 0xD2, 0xA6,
    0xAB, 0xF7, 0x15, 0x88, 0x09, 0xCF, 0x4F, 0x3C
};

#endif

// LEDs and Frame jobs
osjob_t rxLedJob;
osjob_t txLedJob;
osjob_t sendFrameJob;

// LED state
static bool AppLedStateOn = false;

//////////////////////////////////////////////////
// Utility functions
//////////////////////////////////////////////////
/*!
 * \brief Computes a random number between min and max
 *
 * \param [IN] min range minimum value
 * \param [IN] max range maximum value
 * \retval random random value in range min..max
 */
int32_t randr( int32_t min, int32_t max )
{
    return ( int32_t )rand( ) % ( max - min + 1 ) + min;
}

//////////////////////////////////////////////////
// APPLICATION CALLBACKS
//////////////////////////////////////////////////

// provide application router ID (8 bytes, LSBF)
void os_getArtEui( uint8_t *buf )
{
    memcpy( buf, AppEui, 8 );
}

// provide device ID (8 bytes, LSBF)
void os_getDevEui( uint8_t *buf )
{
    memcpy( buf, DevEui, 8 );
}

// provide device key (16 bytes)
void os_getDevKey( uint8_t *buf )
{
    memcpy( buf, DevKey, 16 );
}

//////////////////////////////////////////////////
// MAIN - INITIALIZATION AND STARTUP
//////////////////////////////////////////////////

static void onRxLed( osjob_t* j )
{
    debug_val("LED2 = ", 0 );
}

static void onTxLed( osjob_t* j )
{
    debug_val("LED1 = ", 0 );
}

static void prepareTxFrame( void )
{
    LMIC.frame[0] = AppLedStateOn;
#if ( LORAWAN_CONFIRMED_MSG_ON == 1 )
    LMIC.frame[1] = LMIC.seqnoDn >> 8;
    LMIC.frame[2] = LMIC.seqnoDn;
    LMIC.frame[3] = LMIC.rssi >> 8;
    LMIC.frame[4] = LMIC.rssi;
    LMIC.frame[5] = LMIC.snr;
#endif    
/*    debug_str("Frame to be sent: ");
    debug_val("1: ", LMIC.frame[0]);
    debug_val("2: ", LMIC.frame[1]);
    debug_val("3: ", LMIC.frame[2]);
    debug_val("4: ", LMIC.frame[3]);
    debug_val("5: ", LMIC.frame[4]);
    debug_val("6: ", LMIC.frame[5]);
    debug_str("\r\n");*/
}

void processRxFrame( void )
{
    debug_str("Data: ");
    debug_buf( LMIC.frame + LMIC.dataBeg, LMIC.dataLen );
/*    debug_str("Data - command: ");
    debug_hex(LMIC.frame[0]);
    debug_str("\r\n");*/

    switch( LMIC.frame[LMIC.dataBeg - 1] ) // Check Rx port number
    {
        case 0:
//            debug_str("Port 0!!!\r\n");
//            debug_val("Data Len: ", LMIC.dataLen);
            
        case 1: // The application LED can be controlled on port 1 or 2
        case 2:
            if( LMIC.dataLen == 1 )
            {
                debug_str("Data received on port 2: ");
                debug_hex(LMIC.frame[LMIC.dataBeg]);
                debug_str("\r\n");
                AppLedStateOn = LMIC.frame[LMIC.dataBeg] & 0x01;
                debug_val( "LED3 = ", AppLedStateOn );
            }
            break;
        default:
            break;
    }
}

static void onSendFrame( osjob_t* j )
{
    prepareTxFrame( );

    // Create Registration PDU :
    
    CoapPDU *pdu = new CoapPDU();
    
    pdu->setCode(CoapPDU::COAP_POST);
    pdu->setType(CoapPDU::COAP_CONFIRMABLE);
    int size;
    uint8_t * token = get_Token(&size);
    pdu->setToken(token,size);
    pdu->setMessageID(get_Message_ID());
    pdu->setURI(get_Registration_Query());
    
    _payload=get_Registration_Payload(&_payload_size);
    pdu->setPayload(_payload, (int) _payload_size);
    
    int PDUlength = pdu->getPDULength();
    u1_t frame[PDUlength+6];
    
    memcpy(frame, pdu->getPDUPointer(), PDUlength * sizeof(uint8_t));

    frame[PDUlength] = LMIC.seqnoDn >> 8;
    frame[PDUlength+1] = LMIC.seqnoDn;
    frame[PDUlength+2] = LMIC.rssi >> 8;
    frame[PDUlength+3] = LMIC.rssi;
    frame[PDUlength+4] = LMIC.snr;
    frame[PDUlength+5] = '\0';
    /* debug_str("Frame: ");
    debug_str(pdu->getPDUPointer());
    debug_str("  <STOP>\r\n");

    debug_str("Frame: ");
    debug_str(frame);
    debug_str("  <STOP>\r\n");
    debug_val("Frame Length: ", PDUlength+5);*/
    
    //LMIC_setTxData2( LORAWAN_APP_PORT, frame, PDUlength+6, LORAWAN_CONFIRMED_MSG_ON );
    LMIC_setTxData2( LORAWAN_APP_PORT, LMIC.frame, LORAWAN_APP_DATA_SIZE, LORAWAN_CONFIRMED_MSG_ON );

    // Blink Tx LED
    debug_val( "LED1 = ", 1 );
    os_setTimedCallback( &txLedJob, os_getTime( ) + ms2osticks( 25 ), onTxLed );
}

// Initialization job
static void onInit( osjob_t* j )
{
    // reset MAC state
    LMIC_reset( );
    LMIC_setAdrMode( LORAWAN_ADR_ON );
    LMIC_setDrTxpow( DR_SF12, 14 );

    // start joining
#if( OVER_THE_AIR_ACTIVATION != 0 )
    LMIC_startJoining( );
#else
    LMIC_setSession( LORAWAN_NET_ID, LORAWAN_DEV_ADDR, NwkSKey, ArtSKey );
    onSendFrame( NULL );
#endif
    // init done - onEvent( ) callback will be invoked...
}

int main( void )
{
    debug_init();
    osjob_t initjob;

    // initialize runtime env
    os_init( );
    // setup initial job
    os_setCallback( &initjob, onInit );
    // execute scheduled jobs and events
    os_runloop( );
    // (not reached)
}

//////////////////////////////////////////////////
// LMIC EVENT CALLBACK
//////////////////////////////////////////////////
void onEvent( ev_t ev )
{
    bool txOn = false;
    debug_event( ev );

    switch( ev ) 
    {
    // network joined, session established
    case EV_JOINED:
        debug_val( "Net ID = ", LMIC.netid );
        txOn = true;
        break;
    // scheduled data sent (optionally data received)
    case EV_TXCOMPLETE:
        debug_val( "Datarate = ", LMIC.datarate );
        // Check if we have a downlink on either Rx1 or Rx2 windows
        if( ( LMIC.txrxFlags & ( TXRX_DNW1 | TXRX_DNW2 ) ) != 0 )
        {
            debug_val( "LED2 = ", 1 );
            os_setTimedCallback( &rxLedJob, os_getTime( ) + ms2osticks( 25 ), onRxLed );

            if( LMIC.dataLen != 0 )
            { // data received in rx slot after tx
                //debug_buf( LMIC.frame + LMIC.dataBeg, LMIC.dataLen );
                processRxFrame( );
            }
        }
        txOn = true;
        break;
    default:
        break;
    }
    if( txOn == true )
    {
        //Sends frame every APP_TX_DUTYCYCLE +/- APP_TX_DUTYCYCLE_RND random time (if not duty cycle limited)
        os_setTimedCallback( &sendFrameJob,
                             os_getTime( ) + ms2osticks( APP_TX_DUTYCYCLE + randr( -APP_TX_DUTYCYCLE_RND, APP_TX_DUTYCYCLE_RND ) ),
                             onSendFrame );
        
        ////Sends frame as soon as possible (duty cylce limitations)
        //onSendFrame( NULL );
    }
}
