#include <mbed.h>

// GNSS and Compass test programm for Mateksys GNSS&Compass M9N-5883

#include "Eigen/Dense.h"
#include "QMC5883L.h"
#include "LinearCharacteristics.h"
#include "NEOM9N_thread.h"

bool do_use_calibrated_mag = true;

// logical variable main task
bool do_execute_main_task = false;  // this variable will be toggled via the user button (blue button) to or not to execute the main task

// user button on nucleo board
Timer user_button_timer;            // create Timer object which we use to check if user button was pressed for a certain time (robust against signal bouncing)
InterruptIn user_button(PC_13);     // create InterruptIn interface object to evaluate user button falling and rising edge (no blocking code in ISR)
void user_button_pressed_fcn();     // custom functions which gets executed when user button gets pressed and released, definition below
void user_button_released_fcn();

int main()
{
    // while loop gets executed every main_task_period_ms milliseconds
    const int main_task_period_ms = 50; // define main task period time in ms e.g. 50 ms -> main task runns 20 times per second
    Timer main_task_timer;              // create Timer object which we use to run the main task every main task period time in ms
    main_task_timer.start();
    
    Timer run_timer;
    run_timer.start();

    // led on nucleo board
    DigitalOut user_led(LED1);      // create DigitalOut object to command user led

    // create QMC5883L compass object
    I2C i2c(PB_4, PA_8);
    QMC5883L mag(i2c);
    LinearCharacteristics raw_mx2mx, raw_my2my, raw_mz2mz;
    if (do_use_calibrated_mag) {
        raw_mx2mx.setup(0.9976f, 0.0091f);
        raw_my2my.setup(0.9962f, 0.2145f);
        raw_mz2mz.setup(1.0062f, -0.0920f);
    } else {
        raw_mx2mx.setup(1.0f, 0.0f);
        raw_my2my.setup(1.0f, 0.0f);
        raw_mz2mz.setup(1.0f, 0.0f);
    }
    float mag_val[3] = {0.0f, 0.0f, 0.0f};
    
    // create object for GNSS Sensor NEO-M9N, this runs as an own thread
    NEOM9N neom9n_thread(PC_6, PC_7);
    neom9n_thread.start_loop();
    NEOM9N::ubxNavPVT_t ubxNavPVT; // this is only to access raw data
    bool isGNSSInit = false;
    
    // attach button fall and rise functions to user button object
    user_button.fall(&user_button_pressed_fcn);
    user_button.rise(&user_button_released_fcn);   

    /*
    // dev checksum
    char msg[100] = {0xB5, 0x62, 0x01, 0x07, 0x5C, 0x00, 0xC8, 0x72, 0xAE, 0x16, 0xE6, 0x07, 0x06, 0x09, 0x09, 0x29,
                     0x35, 0xF3, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x84, 0xD7, 0x17, 0x00, 0x00, 0xA4, 0x00, 0x00, 0x00,
                     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x98, 0xBD, 0xFF, 0xFF, 0xFF, 0xFF,
                     0xFF, 0xFF, 0x00, 0x34, 0xF8, 0xDF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x58, 0x3E, 0x0F, 0x00, 0x80, 0xA8,
                     0x12, 0x01, 0x0F, 0x27, 0x00, 0x00, 0xEE, 0x13, 0x4F, 0x2F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                     0x00, 0x00, 0xB9, 0x53};
    uint8_t CK_A = 0x08; // 0x01 + 0x07
    uint8_t CK_B = 0x09; // 0x01 + 0x01 + 0x07;
    printf("%02x %02x\r\n", CK_A, CK_B);
    printf(" ---\r\n");
    bool do_it_once = true;
    while(do_it_once) {
        for (int i = 0; i<100; i++) {
            printf("%02x ", msg[i]);
            if ((i + 1)%16 == 0)printf("\r\n");
            if (i >= 4 && i<=97) {
                CK_A += msg[i];
                CK_B += CK_A;
            }
        }
        printf("\r\n");
        printf(" ---\r\n");
        printf("%02x %02x\r\n", CK_A, CK_B);
        printf(" ---\r\n");
        do_it_once = false;
    }
    */

    while (true) { // this loop will run forever

        main_task_timer.reset();
        //run_timer.reset(); // comment this to get a time reference

        // update magnetometer
        mag.readMag(); // this needs approx 2450 mus
        mag_val[0] = raw_mx2mx.evaluate( mag.magX() );
        mag_val[1] = raw_my2my.evaluate( mag.magY() );
        mag_val[2] = raw_mz2mz.evaluate( mag.magZ() );

        // manual zero GNSS
        if (do_execute_main_task) {
            neom9n_thread.reset_local();
            do_execute_main_task = false;
        }

        int run_elapsed_time_ms = std::chrono::duration_cast<std::chrono::milliseconds>(run_timer.elapsed_time()).count();

        // readout data and sens via serial to putty
        ubxNavPVT = neom9n_thread.GetUbxNavPVT();
        printf("%0.3f, %0.3f, %0.3f, ", //  1:3
            mag_val[0], mag_val[1], mag_val[2]);
        printf("%0.3d, %0.3d, %0.3d, %0.3d, %0.3d, %0.3f, %0.3f, %0.3d, %0.3d, %0.3d, %0.3d, %0.3f, %0.3f, %0.3f,", // 4:17
                neom9n_thread.GetFixType(), neom9n_thread.GetNumSV(), ubxNavPVT.lon, ubxNavPVT.lat, ubxNavPVT.height,
                neom9n_thread.GethAcc(), neom9n_thread.GetvAcc(), ubxNavPVT.velN, ubxNavPVT.velE, ubxNavPVT.velD,
                ubxNavPVT.gSpeed, neom9n_thread.GetHeadMot(), neom9n_thread.GetsAcc(), neom9n_thread.GetHeadAcc());
        static Eigen::Vector3f pos_ecef;
        static Eigen::Vector3f pos_enu;
        static Eigen::Vector3f vel_enu;
        // data below only gets updated if GNSS has fix, more sats than M_MIN_SATS and a new measurement is available
        pos_ecef = neom9n_thread.GetPosECEF();
        pos_enu = neom9n_thread.GetPosENU();
        vel_enu = neom9n_thread.GetVelENU();
        printf("%0.3f, %0.3f, %0.3f, ", // 18:20
            pos_ecef.x(), pos_ecef.y(), pos_ecef.z());
        printf("%0.3f, %0.3f, %0.3f, ", // 21:23
            pos_enu.x(), pos_enu.y(), pos_enu.z());
        printf("%0.3f, %0.3f, %0.3f, ", // 24:26
            vel_enu.x(), vel_enu.y(), vel_enu.z());
        printf("%d", run_elapsed_time_ms);

        printf("%d\r\n", neom9n_thread.GetGPSTimeOfWeek()); // 27

        user_led = !user_led;

        // read timer and make the main thread sleep for the remaining time span (non blocking)
        int main_task_elapsed_time_ms = std::chrono::duration_cast<std::chrono::milliseconds>(main_task_timer.elapsed_time()).count();  
        thread_sleep_for(main_task_period_ms - main_task_elapsed_time_ms);
    }
}

void user_button_pressed_fcn()
{
    user_button_timer.start();
    user_button_timer.reset();
}

void user_button_released_fcn()
{
    // read timer and toggle do_execute_main_task if the button was pressed longer than the below specified time
    int user_button_elapsed_time_ms = std::chrono::duration_cast<std::chrono::milliseconds>(user_button_timer.elapsed_time()).count();
    user_button_timer.stop();
    if (user_button_elapsed_time_ms > 200) {
        do_execute_main_task = !do_execute_main_task;
    }
}