#include <mbed.h>

#include "PM2_Libary.h"
#include "Eigen/Dense.h"

# define M_PI 3.14159265358979323846  // number pi

// logical variable main task
bool do_execute_main_task = false;  // this variable will be toggled via the user button (blue button) to or not to execute the main task

// user button on nucleo board
Timer user_button_timer;            // create Timer object which we use to check if user button was pressed for a certain time (robust against signal bouncing)
InterruptIn user_button(PC_13);     // create InterruptIn interface object to evaluate user button falling and rising edge (no blocking code in ISR)
void user_button_pressed_fcn();     // custom functions which gets executed when user button gets pressed and released, definition below
void user_button_released_fcn();

float ir_distance_mV2cm(float ir_distance_mV);

int main()
{
    // while loop gets executed every main_task_period_ms milliseconds
    const int main_task_period_ms = 10;   // define main task period time in ms e.g. 50 ms -> main task runns 20 times per second
    Timer main_task_timer;                // create Timer object which we use to run the main task every main task period time in ms

    // a coutner
    uint32_t main_task_cntr = 0;

    // led on nucleo board
    DigitalOut user_led(LED1);      // create DigitalOut object to command user led

    // Sharp GP2Y0A41SK0F, 4-40 cm IR Sensor
    float ir_distance_mV = 0.0f;    // define variable to store measurement
    float ir_distance_cm = 0.0f;    // compensated sensor value in cm
    AnalogIn ir_analog_in(PC_2);    // create AnalogIn object to read in infrared distance sensor, 0...3.3V are mapped to 0...1

    // create SensorBar object for sparkfun line follower array, only use this if it is connected (blocking your code if not)
    float sensor_bar_avgAngleRad = 0.0f;
    I2C i2c(PB_9, PB_8);
    //SensorBar sensor_bar(i2c, 0.1175f); // second input argument is distance from bar to wheel axis

    // 78:1, 100:1, ... Metal Gearmotor 20Dx44L mm 12V CB
    DigitalOut enable_motors(PB_15);    // create DigitalOut object to enable dc motors

    FastPWM pwm_M1(PB_13);               // motor M1 is closed-loop speed controlled (angle velocity)
    FastPWM pwm_M2(PA_9);                // motor M2 is closed-loop position controlled (angle controlled)

    EncoderCounter  encoder_M1(PA_6, PC_7); // create encoder objects to read in the encoder counter values
    EncoderCounter  encoder_M2(PB_6, PB_7);

    // create SpeedController and PositionController objects, default parametrization is for 78.125:1 gear box
    const float max_voltage = 12.0f;                  // define maximum voltage of battery packs, adjust this to 6.0f V if you only use one batterypack
    const float counts_per_turn = 20.0f * 78.125f;    // define counts per turn at gearbox end: counts/turn * gearratio
    const float kn = 180.0f / 12.0f;                  // define motor constant in rpm per V
    //const float k_gear = 100.0f / 78.125f;            // define additional ratio in case you are using a dc motor with a different gear box, e.g. 100:1
    //const float kp = 0.1f;                            // define custom kp, this is the default speed controller gain for gear box 78.125:1

    SpeedController speedController_M1(counts_per_turn, kn, max_voltage, pwm_M1, encoder_M1); // default 78.125:1 gear box  with default contoller parameters
    //SpeedController speedController_M1(counts_per_turn * k_gear, kn / k_gear, max_voltage, pwm_M1, encoder_M1); // parameters adjusted to 100:1 gear
    speedController_M1.setMaxAccelerationRPS(999.0f);  // disable internal trajectory planer

    PositionController positionController_M2(counts_per_turn, kn, max_voltage, pwm_M2, encoder_M2); // default 78.125:1 gear with default contoller parameters
    //PositionController positionController_M2(counts_per_turn * k_gear, kn / k_gear, max_voltage, pwm_M2, encoder_M2); // parameters adjusted to 100:1 gear, we need a different speed controller gain here
    //positionController_M2.setSpeedCntrlGain(kp * k_gear);
    positionController_M2.setMaxAccelerationRPS(999.0f);  // disable internal trajectory planer
    // define maximum speed that the position controller is changig the speed, has to be smaller or equal to kn * max_voltage
    float max_speed_rps = 2.0f;
    positionController_M2.setMaxVelocityRPS(max_speed_rps);

    // attach button fall and rise functions to user button object
    user_button.fall(&user_button_pressed_fcn);
    user_button.rise(&user_button_released_fcn);

    // start timer
    main_task_timer.start();

    // enable hardwaredriver dc motors: 0 -> disabled, 1 -> enabled
    enable_motors = 1;

    while (true) { // this loop will run forever

        main_task_timer.reset();
        
        // read analog input
        ir_distance_mV = 1.0e3f * ir_analog_in.read() * 3.3f;
        ir_distance_cm = ir_distance_mV2cm(ir_distance_mV);
        
        // read SensorBar, only use this if it is connected (blocking your code if not)
        //if (sensor_bar.isAnyLedActive()) {
        //    sensor_bar_avgAngleRad = sensor_bar.getAvgAngleRad();
        //}

        if (do_execute_main_task) {

            speedController_M1.setDesiredSpeedRPS(2.0f);
            positionController_M2.setDesiredRotation(3.0f);

        } else {

            speedController_M1.setDesiredSpeedRPS(0.0f);
            positionController_M2.setDesiredRotation(0.0f);
            
        }

        // user_led is switching its state every second
        if ( (main_task_cntr%(1000 / main_task_period_ms) == 0) && (main_task_cntr!=0) ) {
            user_led = !user_led;
        }
        main_task_cntr++;
        
        // do only output via serial what's really necessary (this makes your code slow)
        /*
        printf("IR sensor (mV): %3.3f, IR sensor (cm): %3.3f, SensorBar angle (rad): %3.3f, Speed M1 (rps) %3.3f, Position M2 (rot): %3.3f\r\n",
               ir_distance_mV,
               ir_distance_cm,
               sensor_bar_avgAngleRad,
               speedController_M1.getSpeedRPS(),
               positionController_M2.getRotation());
        */

        // read timer and make the main thread sleep for the remaining time span (non blocking)
        int main_task_elapsed_time_ms = std::chrono::duration_cast<std::chrono::milliseconds>(main_task_timer.elapsed_time()).count();
        thread_sleep_for(main_task_period_ms - main_task_elapsed_time_ms);
    }
}

void user_button_pressed_fcn()
{
    user_button_timer.start();
    user_button_timer.reset();
}

void user_button_released_fcn()
{
    // read timer and toggle do_execute_main_task if the button was pressed longer than the below specified time
    int user_button_elapsed_time_ms = std::chrono::duration_cast<std::chrono::milliseconds>(user_button_timer.elapsed_time()).count();
    user_button_timer.stop();
    if (user_button_elapsed_time_ms > 200) {
        do_execute_main_task = !do_execute_main_task;
    }
}

float ir_distance_mV2cm(float ir_distance_mV)
{
    // defining these variables static makes them persistent within the function
    static float a =    -4.685f; //  (-6.581, -2.79)
    static float c = 3.017e+04f; //  (2.853e+04, 3.181e+04)

    return c/(ir_distance_mV + 1) + a;
}