#include "mlxlinmaster.h"
#include "mbed.h"

LinMaster::LinMaster(PinName txpin, PinName rxpin) : Master(txpin, rxpin)
{
    Master.baud(115200);
    responseIsValid = false;
    responseLength = 0;
    m_baudrate = B19200;
    setBaudrate(B19200);
}
 
int LinMaster::sendCommand(char command, char *msgData, int length)
{
    char numMsgBytes = 0; //# of bytes in message excluding CRC
    switch(command)
    {
        case CMDRESETHARDWARE:
            numMsgBytes = 1;
            break;    
        case CMDHARDWAREID:
            numMsgBytes = 1;
            break;
        case CMDSOFTWAREID:
            numMsgBytes = 1;
            break;
        case CMDCONFIGUREMASTER:
            numMsgBytes = 2;
            break;
        case CMDSENDLINMSG:
            numMsgBytes = length + 5;
            break;
        case CMDREADLASTLINMSG:
            numMsgBytes = 1;
            break;
        case CMDSENDWAKEUP:
            numMsgBytes = 1;
            break;
        default:
            return -1; //command not recognized
    }
    Master.putc(numMsgBytes); //# of bytes
    Master.putc(command); //CMD
    for(int i = 0; i < numMsgBytes-1; i++)
    {
        Master.putc(msgData[i]);    
    }
    Master.putc(calculateChecksum(command, msgData, numMsgBytes-1)); //checksum
    
    int bytesReceived = 0;
    while(Master.readable())
    {
        msgResponse[bytesReceived] = Master.getc();
        bytesReceived++;
    }
    responseLength = bytesReceived;
    
    if(bytesReceived) //check if response from Lin master was received
    {
        char responseLength = msgResponse[0];
        if(msgResponse[1] == command) //response contains correct command
        {           
            if(msgResponse[0] == 0x28)
            {
                responseIsValid = true;
                return 0; //reset of hardware successful but does not allow for CRC check    
            }
            char crcCheck = 0;
            for(int i = 1; i <= responseLength + 1; i++)
            {
                crcCheck += msgResponse[i];    
            }
            if(crcCheck == 255)
            {
                responseIsValid = true;
                return 0; //response was successful   
            }
            else
            {
                responseIsValid = false;
                return -1; //crc check failed
            }
        }
        else
        {
            responseIsValid = false;
            return -1; //invalid response
        }
    }
    else
    {
        responseIsValid = false;
        return -1; //error no response from Lin Master
    }

}

char LinMaster::calculateChecksum(char command, char *msgData, int dataLength)    
{
    int checksum = command;
    for(int i = 0; i < dataLength; i++)
    {
        checksum += msgData[i];
        if(checksum > 255)
            checksum -= 255;    
    }    
    checksum  = 255 - checksum;
    return (char)checksum;
}

char LinMaster::sendOverBus(LinMsg *message)
{
    responseIsValid = false;
    char sendMessage[12] = {0};
    char diagnostic = 0;
    char receiveMessage[8];
    //char length;
    //char parameter;
    //int nbBytes;
    
    switch(message->GetCategory())
    {
        case MSG_S2M:
            sendMessage[0] = 0x80 + message->GetNbBytes() - 1;              //set parameter byte
            sendMessage[1] = 0x55;          //set sync byte
            sendMessage[2] = message->GetIdPar();
            sendMessage[3] = 0x00; //checksum is blank for S2M
            if(sendCommand(CMDSENDLINMSG, sendMessage) == 0)      //check if message sent correctly
            {
                if(sendCommand(CMDREADLASTLINMSG) == 0)        //check if read of bus was successful
                {
                    diagnostic = msgResponse[2];
                    switch(diagnostic)     //check diagnostic byte
                    {
                        case 0x00:     //no erros detected
                            for(int i = 0; i < message->GetNbBytes(); i++)
                            {
                                receiveMessage[i] = msgResponse[i+4];           //read received lin bytes
                            }
                            message->SetDatabytes(receiveMessage);              //move received lin bytes into lin msg buffer
                            message->SetChecksum(message->CalculateChecksum()); //set checksum for new lin bytes
                            break;
                        case 0x80:  //no 12V supply
                            break;
                        case 0x81:  //no 12V supply + no slave response
                            break;
                        case 0x82:  //no 12V supply + busy
                            break;
                        case 0x01:  //no slave response
                            break;
                        case 0x02:  //bus is busy
                            break;
                        default:
                            break;
                    }
                    return diagnostic;
                    
                }
            }

        case MSG_M2S:
            sendMessage[0] = 0x88;              //set parameter byte
            sendMessage[1] = 0x55;          //set sync byte
            sendMessage[2] = message->GetIdPar();
            for(int i = 0; i < message->GetNbBytes(); i++)
            {
                sendMessage[i+3] = message->GetDatabyte(i);    
            }
            sendMessage[11] = message->GetChecksum();
            
            if(sendCommand(CMDSENDLINMSG, sendMessage, message->GetNbBytes()) == 0)      //check if message sent correctly
            {
                if(sendCommand(CMDREADLASTLINMSG) == 0)        //check if read of bus was successful
                {
                    diagnostic = msgResponse[2];
                    switch(diagnostic)     //check diagnostic byte
                    {
                        case 0x00:     //no erros detected
                            break;
                        case 0x80:  //no 12V supply
                            break;
                        case 0x81:  //no 12V supply + no slave response
                            break;
                        case 0x82:  //no 12V supply + busy
                            break;
                        case 0x01:  //no slave response
                            break;
                        case 0x02:  //bus is busy
                            break;
                        default:
                            break;
                    }
                    return diagnostic;
                }
            }
            break;
        case MSG_WKU:
            sendCommand(CMDSENDWAKEUP);
            return 0;
        default:
            break;    
    }       
    return 0; // change
}

void LinMaster::setBaudrate(T_baudrate baudrate)
{
    char sendMessage[1] = {baudrate};
    sendCommand(CMDCONFIGUREMASTER, sendMessage);    
    m_baudrate = baudrate;
}
    
