/***************************************************************************//**
 *   @file   iio_transport.c
 *   @brief  Implementation of iio transport layer interface for AD7606
********************************************************************************
 * Copyright (c) 2020 Analog Devices, Inc.
 *
 * This software is proprietary to Analog Devices, Inc. and its licensors.
 * By using this software you agree to the terms of the associated
 * Analog Devices Software License Agreement.
*******************************************************************************/

/******************************************************************************/
/***************************** Include Files **********************************/
/******************************************************************************/

#include "iio_transport.h"

/******************************************************************************/
/********************** Macros and Constants Definition ***********************/
/******************************************************************************/

/* Max buffer length to hold IIO client data/command */
#define IIO_CMD_DATA_BUFF_SIZE	(100)

/******************************************************************************/
/*************************** Types Declarations *******************************/
/******************************************************************************/

/* UART descriptor to read/write UART peripheral data */
static struct uart_desc *uart_desc = NULL;

/* IIO command buffer */
static volatile char iio_cmd_buffer[IIO_CMD_DATA_BUFF_SIZE];

/* IIO data buffer */
static volatile char iio_data_buffer[IIO_CMD_DATA_BUFF_SIZE];

/* New IIO command detect flag */
static volatile bool new_iio_cmd_detected = false;

/* Data buffer index */
volatile static uint8_t data_buff_indx = 0;

/* Command buffer index */
volatile static uint8_t cmd_buff_indx = 0;

/******************************************************************************/
/************************ Functions Declarations ******************************/
/******************************************************************************/

/******************************************************************************/
/************************ Functions Definitions *******************************/
/******************************************************************************/

/**
 * @brief	IIO Wrapper to initialize the UART peripheral
 * @param	desc- Pointer UART descriptor structure
 * @param	param- Pointer to UART init param structure
 * @return	uart init status (SUCCESS/FAILURE)
 */
int32_t iio_uart_init(struct uart_desc **desc, struct uart_init_param *param)
{
	if (uart_init(desc, param) == SUCCESS) {
		/* Save the descriptor for data read/write operation */
		uart_desc = *desc;

		return SUCCESS;
	} else {
		return FAILURE;
	}
}


/**
 * @brief	IIO Wrapper to wrire data to UART peripheral
 * @param	buf- Pointer to buffer containing data
 * @param	len- Number of bytes to write
 * @return	bytes len in case of success, 0 otherwise
 */
ssize_t iio_uart_write(const char *buf, size_t len)
{
	if (uart_write(uart_desc, (uint8_t *)buf, len) == SUCCESS) {
		return len;
	} else {
		return 0;
	}
}


/**
 * @brief	IIO Wrapper to read iio data/command from UART peripheral
 * @param	buf- Pointer to buffer containing data
 * @param	len- Number of bytes to read
 * @return	bytes len in case of success, 0 otherwise
 */
ssize_t iio_uart_read(char *buf, size_t len)
{
	size_t i = 0;

	if (len > 1) {
		do {
			/* Wait until new data of expected length is received
			 * from UART Rx interrupt event */
		} while (data_buff_indx < len);

		/* Get the data into local buffer (entire data) */
		while (i < len) {
			buf[i] = iio_data_buffer[i];
			i++;
		}
		data_buff_indx = 0;
	} else {
		/* Get the iio command into local buffer (1 character at a time) */
		buf[i] = iio_cmd_buffer[cmd_buff_indx];

		/* Reset buffer index after reading whole command */
		if (iio_cmd_buffer[cmd_buff_indx] == '\n') {
			cmd_buff_indx = 0;
		} else {
			cmd_buff_indx++;
		}
	}

	return len;
}


/*!
 * @brief	Callback function to receive IIO command
 * @return	None
 * @details	This function is an UART Rx ISR registered by IIO application
 *			to invoke upon receipt og new character over UART link.
 */
void iio_uart_rx_callback(void)
{
	uint8_t rx_char;

	if (uart_read(uart_desc, &rx_char, 1) == SUCCESS) {

		iio_data_buffer[data_buff_indx++] = rx_char;

		if (rx_char == '\n') {
			/* The iio command is identified with new line character */
			memcpy(iio_cmd_buffer, iio_data_buffer, data_buff_indx);
			data_buff_indx = 0;
			new_iio_cmd_detected = true;
		}
	}
}


/*!
 * @brief	Function to check for newer IIO command
 * @return	IIO command status (true/false)
 */
bool is_new_iio_command_detected(void)
{
	if (new_iio_cmd_detected) {
		new_iio_cmd_detected = false;
		return true;
	} else {
		return false;
	}
}


/*!
 * @brief	Function to check for expected IIO command
 * @param	cmd_str[in] - Expected IIO command string
 * @param	len[in] - Length of IIO command
 * @return	IIO command status (true/false)
 */
bool check_iio_cmd(const char *cmd_str, uint8_t len)
{
	int ret;

	ret = strncmp(cmd_str, iio_cmd_buffer, len);
	if (ret == 0) {
		/* String matches, return true */
		return true;
	}

	return false;
}
