/***************************************************************************//**
*   @file   iio_ad7606_attr.h
*   @brief  File for AD7606 IIO Attributes
********************************************************************************
* Copyright (c) 2020 Analog Devices, Inc.
*
* All rights reserved.
* This software is proprietary to Analog Devices, Inc. and its licensors.
* By using this software you agree to the terms of the associated
* Analog Devices Software License Agreement.
*******************************************************************************/
#ifndef IIO_AD7606_ATTR_H_
#define IIO_AD7606_ATTR_H_

/******************************************************************************/
/***************************** Include Files **********************************/
/******************************************************************************/

#include <stdio.h>
#include <stdbool.h>

#include "app_config.h"
#include "iio.h"
#include "iio_types.h"

/******************************************************************************/
/****************************** Macros ****************************************/
/******************************************************************************/

extern ssize_t get_chn_scale(void *device, char *buf, size_t len,
			     const struct iio_ch_info *channel);
extern ssize_t set_chn_scale(void *device, char *buf, size_t len,
			     const struct iio_ch_info *channel);

extern ssize_t get_chn_range(void *device, char *buf, size_t len,
			     const struct iio_ch_info *channel);
extern ssize_t set_chn_range(void *device, char *buf, size_t len,
			     const struct iio_ch_info *channel);

extern ssize_t get_chn_gain(void *device, char *buf, size_t len,
			    const struct iio_ch_info *channel);
extern ssize_t set_chn_gain(void *device, char *buf, size_t len,
			    const struct iio_ch_info *channel);

extern ssize_t get_chn_offset(void *device, char *buf, size_t len,
			      const struct iio_ch_info *channel);
extern ssize_t set_chn_offset(void *device, char *buf, size_t len,
			      const struct iio_ch_info *channel);

extern ssize_t get_chn_phase_offset(void *device, char *buf, size_t len,
				    const struct iio_ch_info *channel);
extern ssize_t set_chn_phase_offset(void *device, char *buf, size_t len,
				    const struct iio_ch_info *channel);

extern ssize_t get_chn_temperature(void *device, char *buf, size_t len,
				   const struct iio_ch_info *channel);
extern ssize_t set_chn_temperature(void *device, char *buf, size_t len,
				   const struct iio_ch_info *channel);

extern ssize_t get_chn_vref(void *device, char *buf, size_t len,
			    const struct iio_ch_info *channel);
extern ssize_t set_chn_vref(void *device, char *buf, size_t len,
			    const struct iio_ch_info *channel);

extern ssize_t get_chn_vdrive(void *device, char *buf, size_t len,
			      const struct iio_ch_info *channel);
extern ssize_t set_chn_vdrive(void *device, char *buf, size_t len,
			      const struct iio_ch_info *channel);

extern ssize_t get_chn_aldo(void *device, char *buf, size_t len,
			    const struct iio_ch_info *channel);
extern ssize_t set_chn_aldo(void *device, char *buf, size_t len,
			    const struct iio_ch_info *channel);

extern ssize_t get_chn_dldo(void *device, char *buf, size_t len,
			    const struct iio_ch_info *channel);
extern ssize_t set_chn_dldo(void *device, char *buf, size_t len,
			    const struct iio_ch_info *channel);

extern ssize_t get_bandwidth(void *device, char *buf, size_t len,
			     const struct iio_ch_info *channel);
extern ssize_t set_bandwidth(void *device, char *buf, size_t len,
			     const struct iio_ch_info *channel);

extern ssize_t get_chn_open_wire_detect(void *device, char *buf, size_t len,
					const struct iio_ch_info *channel);
extern ssize_t set_chn_open_wire_detect(void *device, char *buf, size_t len,
					const struct iio_ch_info *channel);

extern ssize_t get_chn_calibrate_adc(void *device, char *buf, size_t len,
				     const struct iio_ch_info *channel);
extern ssize_t set_chn_calibrate_adc(void *device, char *buf, size_t len,
				     const struct iio_ch_info *channel);

extern ssize_t get_chn_raw(void *device, char *buf, size_t len,
			   const struct iio_ch_info *channel);
extern ssize_t set_chn_raw(void *device, char *buf, size_t len,
			   const struct iio_ch_info *channel);

extern ssize_t get_operating_mode(void *device, char *buf, size_t len,
				  const struct iio_ch_info *channel);
extern ssize_t set_operating_mode(void *device, char *buf, size_t len,
				  const struct iio_ch_info *channel);

extern ssize_t get_power_down_mode(void *device, char *buf, size_t len,
				   const struct iio_ch_info *channel);
extern ssize_t set_power_down_mode(void *device, char *buf, size_t len,
				   const struct iio_ch_info *channel);

extern ssize_t get_range(void *device, char *buf, size_t len,
			 const struct iio_ch_info *channel);
extern ssize_t set_range(void *device, char *buf, size_t len,
			 const struct iio_ch_info *channel);

extern ssize_t get_oversampling(void *device, char *buf, size_t len,
				const struct iio_ch_info *channel);
extern ssize_t set_oversampling(void *device, char *buf, size_t len,
				const struct iio_ch_info *channel);

extern ssize_t get_direct_reg_access(void *device, char *buf, size_t len,
				     const struct iio_ch_info *channel);
extern ssize_t set_direct_reg_access(void *device, char *buf, size_t len,
				     const struct iio_ch_info *channel);

/******************************************************************************/
/*************************** Types Declarations *******************************/
/******************************************************************************/

// XML describes the IIO context hierarchy for AD7606 family devices
static const char * const ad7606_phy_xml =
#if defined(DEV_AD7605_4)
	"<device id=\"AD7605-4-phy\" name=\"ID_AD7605-4-phy\" >"
#elif defined(DEV_AD7606_4)
	"<device id=\"AD7606-4-phy\" name=\"AD7606-4-phy\" >"
#elif defined(DEV_AD7606_6)
	"<device id=\"AD7606-6-phy\" name=\"AD7606-6-phy\" >"
#elif defined(DEV_AD7606_8)
	"<device id=\"AD7606-8-phy\" name=\"AD7606-8-phy\" >"
#elif defined(DEV_AD7606B)
	"<device id=\"AD7606B-phy\" name=\"AD7606B-phy\" >"
#elif defined(DEV_AD7606C_16)
	"<device id=\"AD7606C-phy\" name=\"AD7606C-16-phy\" >"
#elif defined(DEV_AD7606C_18)
	"<device id=\"AD7606C-phy\" name=\"AD7606C-18-phy\" >"
#elif defined(DEV_AD7608)
	"<device id=\"AD7608-phy\" name=\"AD7608-phy\" >"
#elif defined(DEV_AD7609)
	"<device id=\"AD7609-phy\" name=\"AD7609-phy\" >"
#else
#endif
	"<channel id=\"voltage1\" type=\"input\" >"
#if (ADC_RESOLUTION > 16)
	"<scan-element scale=\"0.1525\" format=\"le:s18/18>>0\" index =\"0\" />"
#else
	"<scan-element scale=\"0.1525\" format=\"le:s16/16>>0\" index =\"0\" />"
#endif
	"<attribute name=\"raw\" />"
	"<attribute name=\"scale\" />"
#if defined(DEV_AD7606B) || defined(DEV_AD7606C_18) || defined(DEV_AD7606C_16)
	"<attribute name=\"chn_range\" />"
	"<attribute name=\"chn_gain\" />"
	"<attribute name=\"chn_offset\" />"
	"<attribute name=\"Temperature\" />"
	"<attribute name=\"Vref\" />"
	"<attribute name=\"Vdrive\" />"
	"<attribute name=\"ALDO\" />"
	"<attribute name=\"DLDO\" />"
	"<attribute name=\"perform_open_wire_detect\" />"
	"<attribute name=\"do_calibration_(Vin_must_be_0)\" />"
#if defined(DEV_AD7606C_18) || defined(DEV_AD7606C_16)
	"<attribute name=\"bandwidth\" />"
#endif
#endif
	"</channel>"
	"<channel id=\"voltage2\" type=\"input\" >"
#if (ADC_RESOLUTION > 16)
	"<scan-element scale=\"0.1525\" format=\"le:s18/18>>0\" index =\"1\" />"
#else
	"<scan-element scale=\"0.1525\" format=\"le:s16/16>>0\" index =\"1\" />"
#endif
	"<attribute name=\"raw\" />"
	"<attribute name=\"scale\" />"
#if defined(DEV_AD7606B) || defined(DEV_AD7606C_18) || defined(DEV_AD7606C_16)
	"<attribute name=\"chn_range\" />"
	"<attribute name=\"chn_gain\" />"
	"<attribute name=\"chn_offset\" />"
	"<attribute name=\"Temperature\" />"
	"<attribute name=\"Vref\" />"
	"<attribute name=\"Vdrive\" />"
	"<attribute name=\"ALDO\" />"
	"<attribute name=\"DLDO\" />"
	"<attribute name=\"perform_open_wire_detect\" />"
	"<attribute name=\"do_calibration_(Vin_must_be_0)\" />"
#if defined(DEV_AD7606C_18) || defined(DEV_AD7606C_16)
	"<attribute name=\"bandwidth\" />"
#endif
#endif
	"</channel>"
	"<channel id=\"voltage3\" type=\"input\" >"
#if (ADC_RESOLUTION > 16)
	"<scan-element scale=\"0.1525\" format=\"le:s18/18>>0\" index =\"2\" />"
#else
	"<scan-element scale=\"0.1525\" format=\"le:s16/16>>0\" index =\"2\" />"
#endif
	"<attribute name=\"raw\" />"
	"<attribute name=\"scale\" />"
#if defined(DEV_AD7606B) || defined(DEV_AD7606C_18) || defined(DEV_AD7606C_16)
	"<attribute name=\"chn_range\" />"
	"<attribute name=\"chn_gain\" />"
	"<attribute name=\"chn_offset\" />"
	"<attribute name=\"Temperature\" />"
	"<attribute name=\"Vref\" />"
	"<attribute name=\"Vdrive\" />"
	"<attribute name=\"ALDO\" />"
	"<attribute name=\"DLDO\" />"
	"<attribute name=\"perform_open_wire_detect\" />"
	"<attribute name=\"do_calibration_(Vin_must_be_0)\" />"
#if defined(DEV_AD7606C_18) || defined(DEV_AD7606C_16)
	"<attribute name=\"bandwidth\" />"
#endif
#endif
	"</channel>"
	"<channel id=\"voltage4\" type=\"input\" >"
#if (ADC_RESOLUTION > 16)
	"<scan-element scale=\"0.1525\" format=\"le:s18/18>>0\" index =\"3\" />"
#else
	"<scan-element scale=\"0.1525\" format=\"le:s16/16>>0\" index =\"3\" />"
#endif
	"<attribute name=\"raw\" />"
	"<attribute name=\"scale\" />"
#if defined(DEV_AD7606B) || defined(DEV_AD7606C_18) || defined(DEV_AD7606C_16)
	"<attribute name=\"chn_range\" />"
	"<attribute name=\"chn_gain\" />"
	"<attribute name=\"chn_offset\" />"
	"<attribute name=\"Temperature\" />"
	"<attribute name=\"Vref\" />"
	"<attribute name=\"Vdrive\" />"
	"<attribute name=\"ALDO\" />"
	"<attribute name=\"DLDO\" />"
	"<attribute name=\"perform_open_wire_detect\" />"
	"<attribute name=\"do_calibration_(Vin_must_be_0)\" />"
#if defined(DEV_AD7606C_18) || defined(DEV_AD7606C_16)
	"<attribute name=\"bandwidth\" />"
#endif
#endif
	"</channel>"
#if (NO_OF_CHANNELS > 4)
	"<channel id=\"voltage5\" type=\"input\" >"
#if (ADC_RESOLUTION > 16)
	"<scan-element scale=\"0.1525\" format=\"le:s18/18>>0\" index =\"4\" />"
#else
	"<scan-element scale=\"0.1525\" format=\"le:s16/16>>0\" index =\"4\" />"
#endif
	"<attribute name=\"raw\" />"
	"<attribute name=\"scale\" />"
#if defined(DEV_AD7606B) || defined(DEV_AD7606C_18) || defined(DEV_AD7606C_16)
	"<attribute name=\"chn_range\" />"
	"<attribute name=\"chn_oversampling\" />"
	"<attribute name=\"chn_gain\" />"
	"<attribute name=\"chn_offset\" />"
	"<attribute name=\"Temperature\" />"
	"<attribute name=\"Vref\" />"
	"<attribute name=\"Vdrive\" />"
	"<attribute name=\"ALDO\" />"
	"<attribute name=\"DLDO\" />"
	"<attribute name=\"perform_open_wire_detect\" />"
	"<attribute name=\"do_calibration_(Vin_must_be_0)\" />"
#if defined(DEV_AD7606C_18) || defined(DEV_AD7606C_16)
	"<attribute name=\"bandwidth\" />"
#endif
#endif
	"</channel>"
	"<channel id=\"voltage6\" type=\"input\" >"
#if (ADC_RESOLUTION > 16)
	"<scan-element scale=\"0.1525\" format=\"le:s18/18>>0\" index =\"5\" />"
#else
	"<scan-element scale=\"0.1525\" format=\"le:s16/16>>0\" index =\"5\" />"
#endif
	"<attribute name=\"raw\" />"
	"<attribute name=\"scale\" />"
#if defined(DEV_AD7606B) || defined(DEV_AD7606C_18) || defined(DEV_AD7606C_16)
	"<attribute name=\"chn_range\" />"
	"<attribute name=\"chn_gain\" />"
	"<attribute name=\"chn_offset\" />"
	"<attribute name=\"Temperature\" />"
	"<attribute name=\"Vref\" />"
	"<attribute name=\"Vdrive\" />"
	"<attribute name=\"ALDO\" />"
	"<attribute name=\"DLDO\" />"
	"<attribute name=\"perform_open_wire_detect\" />"
	"<attribute name=\"do_calibration_(Vin_must_be_0)\" />"
#if defined(DEV_AD7606C_18) || defined(DEV_AD7606C_16)
	"<attribute name=\"bandwidth\" />"
#endif
#endif
	"</channel>"
#endif
#if (NO_OF_CHANNELS > 6)
	"<channel id=\"voltage7\" type=\"input\" >"
#if (ADC_RESOLUTION > 16)
	"<scan-element scale=\"0.1525\" format=\"le:s18/18>>0\" index =\"6\" />"
#else
	"<scan-element scale=\"0.1525\" format=\"le:s16/16>>0\" index =\"6\" />"
#endif
	"<attribute name=\"raw\" />"
	"<attribute name=\"scale\" />"
#if defined(DEV_AD7606B) || defined(DEV_AD7606C_18) || defined(DEV_AD7606C_16)
	"<attribute name=\"chn_range\" />"
	"<attribute name=\"chn_gain\" />"
	"<attribute name=\"chn_offset\" />"
	"<attribute name=\"Temperature\" />"
	"<attribute name=\"Vref\" />"
	"<attribute name=\"Vdrive\" />"
	"<attribute name=\"ALDO\" />"
	"<attribute name=\"DLDO\" />"
	"<attribute name=\"perform_open_wire_detect\" />"
	"<attribute name=\"do_calibration_(Vin_must_be_0)\" />"
#if defined(DEV_AD7606C_18) || defined(DEV_AD7606C_16)
	"<attribute name=\"bandwidth\" />"
#endif
#endif
	"</channel>"
	"<channel id=\"voltage8\" type=\"input\" >"
#if (ADC_RESOLUTION > 16)
	"<scan-element scale=\"0.1525\" format=\"le:s18/18>>0\" index =\"7\" />"
#else
	"<scan-element scale=\"0.1525\" format=\"le:s16/16>>0\" index =\"7\" />"
#endif
	"<attribute name=\"raw\" />"
	"<attribute name=\"scale\" />"
#if defined(DEV_AD7606B) || defined(DEV_AD7606C_18) || defined(DEV_AD7606C_16)
	"<attribute name=\"chn_range\" />"
	"<attribute name=\"chn_gain\" />"
	"<attribute name=\"chn_offset\" />"
	"<attribute name=\"Temperature\" />"
	"<attribute name=\"Vref\" />"
	"<attribute name=\"Vdrive\" />"
	"<attribute name=\"ALDO\" />"
	"<attribute name=\"DLDO\" />"
	"<attribute name=\"perform_open_wire_detect\" />"
	"<attribute name=\"do_calibration_(Vin_must_be_0)\" />"
#if defined(DEV_AD7606C_18) || defined(DEV_AD7606C_16)
	"<attribute name=\"bandwidth\" />"
#endif
#endif
	"</channel>"
#endif
#if defined(DEV_AD7606B) || defined(DEV_AD7606C_18) || defined(DEV_AD7606C_16)
	"<attribute name=\"operating_mode\" />"
	"<attribute name=\"oversampling\" />"
	"<debug-attribute name =\"direct_reg_access\" />"
#else
	"<attribute name=\"power_down_mode\" />"
	"<attribute name=\"range\" />"
#endif
	"</device>";


/* Channels IIO attribute structures */

static struct iio_attribute iio_attr_chn_scale = {
	.name = "scale",
	.show = get_chn_scale,
	.store = set_chn_scale,
};

static struct iio_attribute iio_attr_chn_raw = {
	.name = "raw",
	.show = get_chn_raw,
	.store = set_chn_raw,
};

static struct iio_attribute iio_attr_chn_range = {
	.name = "chn_range",
	.show = get_chn_range,
	.store = set_chn_range,
};

static struct iio_attribute iio_attr_chn_gain = {
	.name = "chn_gain",
	.show = get_chn_gain,
	.store = set_chn_gain,
};

static struct iio_attribute iio_attr_chn_offset = {
	.name = "chn_offset",
	.show = get_chn_offset,
	.store = set_chn_offset,
};

static struct iio_attribute iio_attr_chn_phase_offset = {
	.name = "chn_phase offset",
	.show = get_chn_phase_offset,
	.store = set_chn_phase_offset,
};

static struct iio_attribute iio_attr_chn_temperature = {
	.name = "Temperature",
	.show = get_chn_temperature,
	.store = set_chn_temperature,
};

static struct iio_attribute iio_attr_chn_vref = {
	.name = "Vref",
	.show = get_chn_vref,
	.store = set_chn_vref,
};

static struct iio_attribute iio_attr_chn_vdrive = {
	.name = "Vdrive",
	.show = get_chn_vdrive,
	.store = set_chn_vdrive,
};

static struct iio_attribute iio_attr_chn_aldo = {
	.name = "ALDO",
	.show = get_chn_aldo,
	.store = set_chn_aldo,
};

static struct iio_attribute iio_attr_chn_dldo = {
	.name = "DLDO",
	.show = get_chn_dldo,
	.store = set_chn_dldo,
};

static struct iio_attribute iio_attr_bandwidth = {
	.name = "bandwidth",
	.show = get_bandwidth,
	.store = set_bandwidth,
};

static struct iio_attribute iio_attr_chn_open_wire_detect = {
	.name = "perform_open_wire_detect",
	.show = get_chn_open_wire_detect,
	.store = set_chn_open_wire_detect,
};

static struct iio_attribute iio_attr_calibrate_adc= {
	.name = "do_calibration_(Vin_must_be_0)",
	.show = get_chn_calibrate_adc,
	.store = set_chn_calibrate_adc,
};


struct iio_attribute *channel_input_attributes[] = {
	&iio_attr_chn_raw,
	&iio_attr_chn_scale,
#if defined(DEV_AD7606B) || defined(DEV_AD7606C_18) || defined(DEV_AD7606C_16)
	&iio_attr_chn_range,
	&iio_attr_chn_gain,
	&iio_attr_chn_offset,
	&iio_attr_chn_phase_offset,
	&iio_attr_chn_temperature,
	&iio_attr_chn_vref,
	&iio_attr_chn_vdrive,
	&iio_attr_chn_aldo,
	&iio_attr_chn_dldo,
	&iio_attr_chn_open_wire_detect,
	&iio_attr_calibrate_adc,
#if defined(DEV_AD7606C_18) || defined(DEV_AD7606C_16)
	&iio_attr_bandwidth
#endif
#endif
};


/* Device IIO attribute structures */

static struct iio_attribute iio_attr_operating_mode = {
	.name = "operating_mode",
	.show = get_operating_mode,
	.store = set_operating_mode,
};

static struct iio_attribute iio_attr_power_down_mode = {
	.name = "power_down_mode",
	.show = get_power_down_mode,
	.store = set_power_down_mode,
};

static struct iio_attribute iio_attr_range = {
	.name = "range",
	.show = get_range,
	.store = set_range,
};

static struct iio_attribute iio_attr_oversampling = {
	.name = "oversampling",
	.show = get_oversampling,
	.store = set_oversampling,
};

static struct iio_attribute iio_attr_direct_reg_access = {
	.name = "direct_reg_access",
	.show = get_direct_reg_access,
	.store = set_direct_reg_access,
};

static struct iio_attribute *global_attributes[] = {
#if defined(DEV_AD7606B) || defined(DEV_AD7606C_18) || defined(DEV_AD7606C_16)
	&iio_attr_operating_mode,
	&iio_attr_oversampling,
	&iio_attr_direct_reg_access,
#else
	&iio_attr_power_down_mode,
	&iio_attr_range,
#endif
};


static struct iio_channel iio_channel_ch1_in = {
	.name = "voltage1",
	.attributes = channel_input_attributes,
	.ch_out = false,
};

static struct iio_channel iio_channel_ch2_in = {
	.name = "voltage2",
	.attributes = channel_input_attributes,
	.ch_out = false,
};

static struct iio_channel iio_channel_ch3_in = {
	.name = "voltage3",
	.attributes = channel_input_attributes,
	.ch_out = false,
};

static struct iio_channel iio_channel_ch4_in = {
	.name = "voltage4",
	.attributes = channel_input_attributes,
	.ch_out = false,
};

static struct iio_channel iio_channel_ch5_in = {
	.name = "voltage5",
	.attributes = channel_input_attributes,
	.ch_out = false,
};

static struct iio_channel iio_channel_ch6_in = {
	.name = "voltage6",
	.attributes = channel_input_attributes,
	.ch_out = false,
};

static struct iio_channel iio_channel_ch7_in = {
	.name = "voltage7",
	.attributes = channel_input_attributes,
	.ch_out = false,
};

static struct iio_channel iio_channel_ch8_in = {
	.name = "voltage8",
	.attributes = channel_input_attributes,
	.ch_out = false,
};


static struct iio_channel *iio_ad7606_channels[] = {
	&iio_channel_ch1_in,
	&iio_channel_ch2_in,
	&iio_channel_ch3_in,
	&iio_channel_ch4_in,
#if (NO_OF_CHANNELS > 4)
	&iio_channel_ch5_in,
	&iio_channel_ch6_in,
#endif
#if (NO_OF_CHANNELS > 6)
	&iio_channel_ch7_in,
	&iio_channel_ch8_in,
#endif
};

/******************************************************************************/
/************************ Functions Declarations ******************************/
/******************************************************************************/

#endif /* IIO_AD7606_ATTR_H_ */
