/***************************************************************************//**
 * @file  uart.cpp
 * @brief Implementation of UART No-OS platform driver interfaces
********************************************************************************
 * Copyright (c) 2020 Analog Devices, Inc.
 *
 * All rights reserved.
 *
 * This software is proprietary to Analog Devices, Inc. and its licensors.
 * By using this software you agree to the terms of the associated
 * Analog Devices Software License Agreement.
*******************************************************************************/

/******************************************************************************/
/***************************** Include Files **********************************/
/******************************************************************************/
#include <stdio.h>
#include <mbed.h>

#include "platform_drivers.h"
#include "uart_extra.h"

/******************************************************************************/
/************************ Functions Definitions *******************************/
/******************************************************************************/

/**
 * @brief Read data from UART device.
 * @param desc - Instance of UART.
 * @param data - Pointer to buffer containing data.
 * @param bytes_number - Number of bytes to read.
 * @return SUCCESS in case of success, FAILURE otherwise.
 */
int32_t uart_read(struct uart_desc *desc, uint8_t *data, uint32_t bytes_number)
{
	mbed::BufferedSerial *uart;		// pointer to BufferedSerial/UART instance

	if (desc) {
		if (((mbed_uart_desc *)(desc->extra))->uart_port) {
			uart = (BufferedSerial *)(((mbed_uart_desc *)(desc->extra))->uart_port);

			if (data) {
				return uart->read(data, bytes_number);
			}
		}
	}

	return FAILURE;
}


/**
 * @brief Write data to UART device.
 * @param desc - Instance of UART.
 * @param data - Pointer to buffer containing data.
 * @param bytes_number - Number of bytes to read.
 * @return SUCCESS in case of success, FAILURE otherwise.
 */
int32_t uart_write(struct uart_desc *desc, const uint8_t *data,
		   uint32_t bytes_number)
{
	mbed::BufferedSerial *uart;		// pointer to BufferedSerial/UART instance

	if (desc) {
		if (((mbed_uart_desc *)(desc->extra))->uart_port) {
			uart = (BufferedSerial *)(((mbed_uart_desc *)(desc->extra))->uart_port);

			if (data) {
				return uart->write(data, bytes_number);
			}
		}
	}

	return FAILURE;
}


/**
 * @brief Submit reading buffer to the UART driver.
 *
 * Buffer is used until bytes_number bytes are read.
 * @param desc:	Descriptor of the UART device
 * @param data:	Buffer where data will be read
 * @param bytes_number:	Number of bytes to be read.
 * @return \ref SUCCESS in case of success, \ref FAILURE otherwise.
 */
int32_t uart_read_nonblocking(struct uart_desc *desc,
			      uint8_t *data,
			      uint32_t bytes_number)
{
	mbed::BufferedSerial *uart;		// pointer to BufferedSerial/UART instance

	if (desc) {
		if (((mbed_uart_desc *)(desc->extra))->uart_port) {
			uart = (BufferedSerial *)(((mbed_uart_desc *)(desc->extra))->uart_port);

			if (data) {
				for (size_t i = 0; i < bytes_number; i++) {
					if (uart->readable() > 0) {
						uart->read(&data[i], 1);
					}
				}

				return bytes_number;
			}
		}
	}

	return FAILURE;
}


/**
 * @brief Submit writting buffer to the UART driver.
 *
 * Data from the buffer is sent over the UART, the function returns imediatly.
 * @param desc:	Descriptor of the UART device
 * @param data:	Buffer where data will be written
 * @param bytes_number:	Number of bytes to be written.
 * @return \ref SUCCESS in case of success, \ref FAILURE otherwise.
 */
int32_t uart_write_nonblocking(struct uart_desc *desc,
			       const uint8_t *data,
			       uint32_t bytes_number)
{
	mbed::BufferedSerial *uart;		// pointer to BufferedSerial/UART instance

	if (desc) {
		if (((mbed_uart_desc *)(desc->extra))->uart_port) {
			uart = (BufferedSerial *)(((mbed_uart_desc *)(desc->extra))->uart_port);

			if (data) {
				for (size_t i = 0; i < bytes_number; i++) {
					uart->write(&data[i], 1);
				}

				return bytes_number;
			}
		}
	}

	return FAILURE;
}


/**
 * @brief Initialize the UART communication peripheral.
 * @param desc - The UART descriptor.
 * @param param - The structure that contains the UART parameters.
 * @return SUCCESS in case of success, FAILURE otherwise.
 */
int32_t uart_init(struct uart_desc **desc, struct uart_init_param *param)
{
	mbed::BufferedSerial *uart;	// pointer to new BufferedSerial/UART instance
	mbed_uart_desc *mbed_desc; 	// Pointer to mbed uart descriptor

	if (desc && param) {
		// Create the UART description object for the device
		uart_desc *new_desc = (uart_desc *)malloc(sizeof(uart_desc));
		if (new_desc == NULL) {
			return FAILURE;
		}

		new_desc->baud_rate = param->baud_rate;

		// Create and configure a new instance of BufferedSerial/UART port
		uart = new BufferedSerial(
			(PinName)(((mbed_uart_init_param *)param->extra)->uart_tx_pin),
			(PinName)(((mbed_uart_init_param *)param->extra)->uart_rx_pin),
			(int)param->baud_rate);

		if (uart == NULL) {
			return FAILURE;
		}

		// Create a new mbed descriptor to store new UART instances
		mbed_desc = (mbed_uart_desc *)malloc(sizeof(mbed_uart_desc));
		if (mbed_desc == NULL) {
			return FAILURE;
		}

		mbed_desc->uart_port = (BufferedSerial *)uart;
		new_desc->extra = (mbed_uart_desc *)mbed_desc;

		*desc = new_desc;

		return SUCCESS;
	}

	return FAILURE;
}


/**
 * @brief Free the resources allocated by uart_init().
 * @param desc - The UART descriptor.
 * @return SUCCESS in case of success, FAILURE otherwise.
 */
int32_t uart_remove(struct uart_desc *desc)
{
	if (desc) {
		// Free the UART port object
		if ((BufferedSerial *)(((mbed_uart_desc *)(desc->extra))->uart_port)) {
			delete((BufferedSerial *)(((mbed_uart_desc *)(desc->extra))->uart_port));
		}

		// Free the UART extra descriptor object
		if ((mbed_uart_desc *)(desc->extra)) {
			free((mbed_uart_desc *)(desc->extra));
		}

		// Free the UART descriptor object
		free(desc);

		return SUCCESS;
	}

	return FAILURE;
}


/**
 * @brief Get number of UART errors.
 * @param desc - The UART descriptor.
 * @return number of errors.
 */
uint32_t uart_get_errors(struct uart_desc *desc)
{
	if (desc) {
		// Unused variable - fix compiler warning
	}

	return SUCCESS;
}