/*!
 *****************************************************************************
  @file:  adi_console_menu.h

  @brief:   A simple console menu manager handler

  @details:
 -----------------------------------------------------------------------------
 Copyright (c) 2019, 2020 Analog Devices, Inc.
 All rights reserved.

 This software is proprietary to Analog Devices, Inc. and its licensors.
 By using this software you agree to the terms of the associated
 Analog Devices Software License Agreement.

*****************************************************************************/

#ifndef ADI_CONSOLE_MENU_H_
#define ADI_CONSOLE_MENU_H_

#include <stdbool.h>
#include <stdint.h>

#define MENU_ESCAPED            -1
#define MENU_CONTINUE           0
#define MENU_DONE               1

#define ESCAPE_KEY_CODE         (char)0x1B

#define EOL "\r\n"

#ifndef ARRAY_SIZE
#define ARRAY_SIZE(x) ((sizeof (x)) / (sizeof ((x)[0])))
#endif

/* Type Definitions */
// Each menu item is defined by this struct
typedef struct {
	// String displayed for menu item
	char * text;
	// character that can be pressed to select menu item
	char  shortcutKey;
	// Function to be called when menu item is selected, if NULL, no function is called
	int32_t (*action)(uint32_t option);
	// id value passed as the option value when calling menuAction
	uint32_t id;
} console_menu_item;

// This defines a complete menu with items
typedef struct {
	// String to be displayed as the menu title
	char * title;
	// Array of all the menu items
	console_menu_item * items;
	// Number of menuItems
	uint8_t itemCount;
	// Function alled before Menu title is displayed if defined
	void (*headerItem)(void);
	// Function called after menu items are displayed if defined
	void (*footerItem)(void);
	// Should the escape key to exit the menu be enabled?
	bool enableEscapeKey;
} console_menu;

/* Function Declarations */
/* Display a console menu, and handle user interactions */
int32_t adi_do_console_menu(const console_menu * menu);
int32_t adi_get_decimal_int(uint8_t input_len);
uint32_t adi_get_hex_integer(uint8_t input_len);
float adi_get_decimal_float(uint8_t input_len);
void adi_clear_console(void);
void adi_press_any_key_to_continue(void);

#endif /* ADI_CONSOLE_MENU_H_ */
