/***************************************************************************//**
 *   @file   i2c.h
 *   @brief  Header file of I2C Interface
 *   @author DBogdan (dragos.bogdan@analog.com)
********************************************************************************
 * Copyright 2019(c) Analog Devices, Inc.
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *  - Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *  - Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *  - Neither the name of Analog Devices, Inc. nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *  - The use of this software may or may not infringe the patent rights
 *    of one or more patent holders.  This license does not release you
 *    from the requirement that you obtain separate licenses from these
 *    patent holders to use this software.
 *  - Use of the software either in source or binary form, must be run
 *    on or directly connected to an Analog Devices Inc. component.
 *
 * THIS SOFTWARE IS PROVIDED BY ANALOG DEVICES "AS IS" AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, NON-INFRINGEMENT,
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL ANALOG DEVICES BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, INTELLECTUAL PROPERTY RIGHTS, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*******************************************************************************/

#ifndef I2C_H_
#define I2C_H_

/******************************************************************************/
/***************************** Include Files **********************************/
/******************************************************************************/

#include <stdint.h>

/******************************************************************************/
/*************************** Types Declarations *******************************/
/******************************************************************************/

/**
 * @struct i2c_platform_ops
 * @brief Structure holding I2C function pointers that point to the platform
 * specific function
 */
struct i2c_platform_ops ;

/**
 * @struct i2c_init_param
 * @brief Structure holding the parameters for I2C initialization.
 */
typedef struct i2c_init_param {
	/** I2C maximum transfer speed supported */
	uint32_t	max_speed_hz;
	/** Slave address */
	uint8_t		slave_address;
	/** I2C platform specific functions */
	const struct i2c_platform_ops *platform_ops;
	/** I2C extra parameters (device specific parameters) */
	void		*extra;
} i2c_init_param;

/**
 * @struct i2c_desc
 * @brief Structure holding I2C descriptor
 */
typedef struct i2c_desc {
	/** I2C maximum transfer speed supported */
	uint32_t	max_speed_hz;
	/** Slave address */
	uint8_t		slave_address;
	/** I2C platform specific functions */
	const struct i2c_platform_ops *platform_ops;
	/** I2C extra parameters (device specific parameters) */
	void		*extra;
} i2c_desc;

/**
 * @struct i2c_platform_ops
 * @brief Structure holding i2c function pointers that point to the platform
 * specific function
 */
struct i2c_platform_ops {
	/** i2c initialization function pointer */
	int32_t (*i2c_ops_init)(struct i2c_desc **, const struct i2c_init_param *);
	/** i2c write function pointer */
	int32_t (*i2c_ops_write)(struct i2c_desc *, uint8_t *, uint8_t, uint8_t);
	/** i2c write function pointer */
	int32_t (*i2c_ops_read)(struct i2c_desc *, uint8_t *, uint8_t, uint8_t);
	/** i2c remove function pointer */
	int32_t (*i2c_ops_remove)(struct i2c_desc *);
};

/******************************************************************************/
/************************ Functions Declarations ******************************/
/******************************************************************************/

/* Initialize the I2C communication peripheral. */
int32_t i2c_init(struct i2c_desc **desc,
		 const struct i2c_init_param *param);

/* Free the resources allocated by i2c_init(). */
int32_t i2c_remove(struct i2c_desc *desc);

/* Write data to a slave device. */
int32_t i2c_write(struct i2c_desc *desc,
		  uint8_t *data,
		  uint8_t bytes_number,
		  uint8_t stop_bit);

/* Read data from a slave device. */
int32_t i2c_read(struct i2c_desc *desc,
		 uint8_t *data,
		 uint8_t bytes_number,
		 uint8_t stop_bit);

#endif // I2C_H_
