/*
    Copyright (c) 2011 Andy Kirkham
 
    Permission is hereby granted, free of charge, to any person obtaining a copy
    of this software and associated documentation files (the "Software"), to deal
    in the Software without restriction, including without limitation the rights
    to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
    copies of the Software, and to permit persons to whom the Software is
    furnished to do so, subject to the following conditions:
 
    The above copyright notice and this permission notice shall be included in
    all copies or substantial portions of the Software.
 
    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
    OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
    THE SOFTWARE.
*/

#ifndef AJK_STCALLBACK_H
#define AJK_STCALLBACK_H

namespace AjK {

class SimpleTask;
class STcallbackDummy;

/** STcallback - Adds callbacks that take SimpleTask * pointer data type.
 *
 * The Mbed library supplies a callback using the FunctionPointer object as 
 * defined in FunctionPointer.h  However, this callback system does not allow
 * the caller to pass a value to the callback. Likewise, the callback itself
 * cannot return a value.
 *
 * Note, when passing pointers to variables to the callback, if the callback
 * function/method changes that variable's value then it will also change the
 * value the caller sees. If C pointers are new to you, you are strongly
 * advised to read up on the subject. It's pointers that often get beginners
 * into trouble when mis-used.
 *
 * @see http://mbed.org/handbook/C-Data-Types
 * @see http://mbed.org/projects/libraries/svn/mbed/trunk/FunctionPointer.h
 * @see http://mbed.org/cookbook/FunctionPointer
 * @see http://mbed.org/cookbook/FPointer
 */
class STcallback {

protected:

    //! C callback function pointer.
    void (*c_callback)(SimpleTask *); 
    
    //! C++ callback object/method pointer (the object part).
    STcallbackDummy *obj_callback;
    
    //! C++ callback object/method pointer (the method part).
    void (STcallbackDummy::*method_callback)(SimpleTask *);

public:
    
    /** Constructor
     */
    STcallback() {
        c_callback      = NULL;
        obj_callback    = NULL;
        method_callback = NULL;
    }
    
    /** attach - Overloaded attachment function.
     *
     * Attach a C type function pointer as the callback.
     *
     * Note, the callback function prototype must be:-
     * @code
     * void myCallbackFunction(SimpleTask *p);
     * @endcode
     * @param A C function pointer to call.
     */
    void attach(void (*function)(SimpleTask *) = 0) { c_callback = function; }
    
    /** attach - Overloaded attachment function.
     *
     * Attach a C++ type object/method pointer as the callback.
     *
     * Note, the callback method prototype must be:-
     * @code
     *     public:
     *         void myCallbackFunction(SimpleTask *p);
     * @endcode
     * @param A C++ object pointer.
     * @param A C++ method within the object to call.
     */
    template<class T> 
    void attach(T* item = 0, void (T::*method)(SimpleTask *) = 0) { 
        obj_callback    = (STcallbackDummy *)item; 
        method_callback = (void (STcallbackDummy::*)(SimpleTask *))method; 
    }

    /** call - Overloaded callback initiator.
     *
     * call the callback function.
     *
     * @param SimpleTask * pointer.
     */
    void call(SimpleTask *arg) {
        if (c_callback != 0) {
            (*c_callback)(arg);
        }
        else {
            if (obj_callback != 0 && method_callback != 0) {
                (obj_callback->*method_callback)(arg);
            }
        }
    }   
};

}; // namespace AjK ends

using namespace AjK;

#endif
